<?php

namespace Drop\GELProximity\Observer;

use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Api\OrderStatusInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Logger\Logger;
use Drop\GELProximity\Model\Carrier\GELProximity;
use Drop\GELProximity\Model\GelShipmentFactory;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;

/**
 * Class SaveRmaShipment
 * @package Drop\GELProximity\Observer
 * @observer
 */
class SaveRmaShipment implements ObserverInterface
{
    /**
     * @var OrderRepositoryInterface
     */
    protected $orderRepository;

    /**
     * @var GelShipmentRepositoryInterface
     */
    protected $gelShipmentRepository;

    /**
     * @var GelShipmentFactory
     */
    protected $gelShipmentFactory;

    /**
     * @var CustomerSession
     */
    protected $customerSession;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * SaveRmaShipment constructor.
     * @param OrderRepositoryInterface $orderRepository
     * @param GelShipmentRepositoryInterface $gelShipmentRepository
     * @param GelShipmentFactory $gelShipmentFactory
     * @param CustomerSession $customerSession
     * @param Data $helper
     */
    public function __construct(
        OrderRepositoryInterface $orderRepository,
        GelShipmentRepositoryInterface $gelShipmentRepository,
        GelShipmentFactory $gelShipmentFactory,
        CustomerSession $customerSession,
        Data $helper
    ) {
        $this->orderRepository = $orderRepository;
        $this->gelShipmentRepository = $gelShipmentRepository;
        $this->gelShipmentFactory = $gelShipmentFactory;
        $this->customerSession = $customerSession;
        $this->helper = $helper;
    }

    /**
     * Observer method used to save the gel shipment entity when the customer submits a new RMA
     *
     * {@inheritDoc}
     */
    public function execute(Observer $observer): void
    {
        if ($this->helper->isModuleEnabled()) {
            $orderId = $this->customerSession->getData('gel_rma_order_id');
            $shipmentData = $this->customerSession->getData('gel_rma_pickup_point_data');
            if ($orderId !== null && $shipmentData !== null) {
                /** @var Order $order */
                $order = $this->orderRepository->get($orderId);
                //Debug
                $this->helper->logDebug('[OBSERVER_SAVERMASHIPMENT] Saving new RMA shipment entity for order ' . $order->getId());
                $gelShipment = $this->gelShipmentFactory
                    ->create()
                    ->setPickupPointId($shipmentData['pickupPointId'])
                    ->setCost($shipmentData['cost'])
                    ->setCurrency($shipmentData['currencyCost'])
                    ->setIsReturn(true)
                    ->setAdditionalPickupPointData(
                        $this->getAdditionalPickupPointData($shipmentData)
                    )
                    ->setQuoteReference($shipmentData['reference'])
                    ->setQuoteId($order->getQuoteId())
                    ->setOrderId($orderId);
                try {
                    $this->gelShipmentRepository->save($gelShipment);
                } catch (AlreadyExistsException $e) {
                    $this->helper->log(
                        sprintf(
                            '--- [OBSERVER_SAVERMASHIPMENT] Error found when creating GEL shipment entity for order ID %s -> Error: %s',
                            $orderId,
                            $e->getMessage()
                        ),
                        Logger::CRITICAL
                    );
                }
                //Debug
                $this->helper->logDebug('--- [OBSERVER_SAVERMASHIPMENT] Cleaning the customer session.');
                $this->customerSession->setData('gel_rma_pickup_point_data', null);
                $this->customerSession->setData('gel_rma_order_id', null);
            } else {
                //Debug
                $this->helper->logDebug('[OBSERVER_SAVERMASHIPMENT] RMA not placed using GEL Proximity: skipping.');
            }
        } else {
            //Debug
            $this->helper->logDebug('[OBSERVER_SAVERMASHIPMENT] Module not enabled: skipping.');
        }
    }

    /**
     * Retrieves all the additional pickup point data from the GEL callback response and serialize it into a JSON
     *
     * @param string[] $data
     * @return string
     */
    protected function getAdditionalPickupPointData(array $data): string
    {
        $additionalData = [
            'code' => $data['code'],
            'description' => $data['description'],
            'address' => $data['address'],
            'city' => $data['city'],
            'department' => $data['department'],
            'zipCode' => $data['zipCode'],
            'country' => $data['country'],
        ];
        return $this->helper->serializeArray($additionalData);
    }
}
