<?php

namespace Drop\GELProximity\Setup;

use Drop\GELProximity\Api\OrderStatusInterface;
use Exception;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Setup\ModuleContextInterface;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Framework\Setup\UpgradeDataInterface;
use Magento\Sales\Model\Order\StatusFactory;
use Magento\Sales\Model\ResourceModel\Order\Status as StatusResource;
use Symfony\Component\Console\Output\ConsoleOutput;

/**
 * Class UpgradeData
 * @package Drop\GELProximity\Setup
 */
class UpgradeData implements UpgradeDataInterface
{
    /**
     * @var StatusFactory
     */
    protected $orderStatusFactory;

    /**
     * @var StatusResource
     */
    protected $orderStatusResource;

    /**
     * @var ConsoleOutput
     */
    protected $consoleOutput;

    /**
     * UpgradeData constructor.
     * @param StatusFactory $orderStatusFactory
     * @param StatusResource $orderStatusResource
     * @param ConsoleOutput $consoleOutput
     */
    public function __construct(
        StatusFactory $orderStatusFactory,
        StatusResource $orderStatusResource,
        ConsoleOutput $consoleOutput
    ) {
        $this->orderStatusFactory = $orderStatusFactory;
        $this->orderStatusResource = $orderStatusResource;
        $this->consoleOutput = $consoleOutput;
    }

    /**
     * {@inheritDoc}
     */
    public function upgrade(ModuleDataSetupInterface $setup, ModuleContextInterface $context)
    {
        $setup->startSetup();

        if (version_compare($context->getVersion(), '1.1.1') < 0) {
            $this->addSentToGELStatus();
            $this->addShippedStatus();
        }

        $setup->endSetup();
    }

    /**
     * Creates the new status Sent to GEL
     */
    protected function addSentToGELStatus(): void
    {
        //Create and set data to the new status
        $status = $this->orderStatusFactory->create();
        $status->setStatus(OrderStatusInterface::ORDER_STATUS_SENT_TO_GEL);
        $status->setData('label', OrderStatusInterface::ORDER_STATUS_SENT_TO_GEL_LABEL);
        try {
            //Save the new status
            $this->orderStatusResource->save($status);
            //Assign existing state
            $status->assignState(OrderStatusInterface::ORDER_STATE_PROCESSING_GEL, true, true);
        } catch (AlreadyExistsException $e) {
            $this->consoleOutput->writeln(
                sprintf(
                    'The status %s already exists. Skipping.',
                    OrderStatusInterface::ORDER_STATUS_SENT_TO_GEL
                )
            );
        } catch (Exception $e) {
            $this->consoleOutput->writeln(
                sprintf(
                    'There was a problem creating the status %s -> %s',
                    OrderStatusInterface::ORDER_STATUS_SENT_TO_GEL,
                    $e->getMessage()
                )
            );
        }
    }

    /**
     * Creates the new status Shipped
     */
    protected function addShippedStatus(): void
    {
        //Create and set data to the new status
        $status = $this->orderStatusFactory->create();
        $status->setStatus(OrderStatusInterface::ORDER_STATUS_SHIPPED_GEL);
        $status->setData('label', OrderStatusInterface::ORDER_STATUS_SHIPPED_GEL_LABEL);
        try {
            //Save the new status
            $this->orderStatusResource->save($status);
            //Assign existing state
            $status->assignState(OrderStatusInterface::ORDER_STATE_SHIPPED_GEL, true, true);
        } catch (AlreadyExistsException $e) {
            $this->consoleOutput->writeln(
                sprintf(
                    'The status %s already exists. Skipping.',
                    OrderStatusInterface::ORDER_STATUS_SHIPPED_GEL
                )
            );
        } catch (Exception $e) {
            $this->consoleOutput->writeln(
                sprintf(
                    'There was a problem creating the status %s -> %s',
                    OrderStatusInterface::ORDER_STATUS_SHIPPED_GEL,
                    $e->getMessage()
                )
            );
        }
    }
}
