<?php

namespace Drop\GELProximity\Service;

use Drop\GELProximity\Api\ConfigPathInterface;
use Drop\GELProximity\Api\Data\GelShipmentInterface;
use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Api\Service\GelGatewayInterface;
use Drop\GELProximity\Api\Service\Processors\OrderProcessorInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Helper\Email;
use Drop\GELProximity\Logger\Logger;
use Drop\GELProximity\Model\ResourceModel\GelShipment\CollectionFactory as GelShipmentCollectionFactory;
use Exception;
use InvalidArgumentException;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;
use Mirasvit\Rma\Api\Data\ItemInterface;
use Mirasvit\Rma\Api\Data\RmaInterface;
use Mirasvit\Rma\Api\Data\StatusInterface;
use Mirasvit\Rma\Model\ResourceModel\Rma\CollectionFactory as RmaCollectionFactory;
use Mirasvit\Rma\Model\Rma;
use Mirasvit\Rma\Repository\RmaRepository;
use Mirasvit\Rma\Repository\StatusRepository;

/**
 * Class ShipReturnsService
 * @package Drop\GELProximity\Service
 */
class ShipReturnsService extends AbstractService
{
    /**
     * @var Email
     */
    protected $emailHelper;

    /**
     * @var RmaCollectionFactory
     */
    protected $rmaCollectionFactory;

    /**
     * @var OrderRepositoryInterface
     */
    protected $orderRepository;

    /**
     * @var RmaRepository
     */
    protected $rmaRepository;

    /**
     * @var StatusRepository
     */
    protected $rmaStatusRepository;

    /**
     * @var GelShipmentCollectionFactory
     */
    protected $gelShipmentCollectionFactory;

    /**
     * ShipReturnsService constructor.
     * @param Data $helper
     * @param GelShipmentRepositoryInterface $gelShipmentRepository
     * @param GelGatewayInterface $gateway
     * @param OrderProcessorInterface $orderProcessor
     * @param Email $emailHelper
     * @param RmaCollectionFactory $rmaCollectionFactory
     * @param OrderRepositoryInterface $orderRepository
     * @param RmaRepository $rmaRepository
     * @param StatusRepository $rmaStatusRepository
     * @param GelShipmentCollectionFactory $gelShipmentCollectionFactory
     */
    public function __construct(
        Data $helper,
        GelShipmentRepositoryInterface $gelShipmentRepository,
        GelGatewayInterface $gateway,
        OrderProcessorInterface $orderProcessor,
        Email $emailHelper,
        RmaCollectionFactory $rmaCollectionFactory,
        OrderRepositoryInterface $orderRepository,
        RmaRepository $rmaRepository,
        StatusRepository $rmaStatusRepository,
        GelShipmentCollectionFactory $gelShipmentCollectionFactory
    ) {
        parent::__construct($helper, $gelShipmentRepository, $gateway, $orderProcessor);
        $this->emailHelper = $emailHelper;
        $this->rmaCollectionFactory = $rmaCollectionFactory;
        $this->orderRepository = $orderRepository;
        $this->rmaRepository = $rmaRepository;
        $this->rmaStatusRepository = $rmaStatusRepository;
        $this->gelShipmentCollectionFactory = $gelShipmentCollectionFactory;
    }

    /**
     * {@inheritDoc}
     */
    public function run(): int
    {
        if ($this->canBeExecuted()) {
            //Clear duplicates rma
            $this->clearDuplicatesRmas();

            //Init rma collection
            $rmaCollection = $this->initServiceCollection();

            //Debug
            $this->helper->logDebug('[SHIP_RETURNS_SERVICE] Starting cron: found ' . $rmaCollection->getSize() . ' returns to be processed.');
            /** @var Rma $rma */
            foreach ($rmaCollection as $rma) {
                //Send data order to GEL server
                try {
                    $gelResponse = $this->processReturn($rma);
                } catch (InvalidArgumentException $e) {
                    //Catch error in case the deserialization fails
                    $this->helper->log(
                        sprintf(
                            '--- [SHIP_RETURNS_SERVICE] Error for return ID %s during JSON deserialization -> Error: %s',
                            $rma->getId(),
                            $e->getMessage()
                        ),
                        Logger::CRITICAL
                    );
                    continue;
                }
                //Check for success GEL response
                if (array_key_exists('success', $gelResponse) && $gelResponse['success']) {
                    try {
                        //Save external order ID reference inside shipment
                        $this->updateGelShipment($rma, $gelResponse);
                        //Change the return status into Approved
                        $this->updateRma($rma);
                    } catch (Exception $e) {
                        $this->helper->log(
                            sprintf(
                                '--- [SHIP_RETURNS_SERVICE] Error saving gel shipment entity for Return %s -> Error: %s',
                                $rma->getId(),
                                $e->getMessage()
                            ),
                            Logger::ERROR
                        );
                    }
                } else {
                    //Log gel response error
                    $this->helper->log('--- [SHIP_SERVICE] GEL responded with an error: ' . $gelResponse['message'], Logger::CRITICAL);
                    //Send email error
                    $this->sendErrorReport($rma->getData('gel_order_id'), $gelResponse['message']);
                }
            }
            return $rmaCollection->count();
        } else {
            //Debug
            $this->helper->logDebug('[SHIP_RETURNS_SERVICE] Service not enabled: skipping.');
            return 0;
        }
    }

    /**
     * {@inheritDoc}
     */
    protected function initServiceCollection(): AbstractCollection
    {
        //Retrieve the status ID of the returns to be processed (configured in admin panel)
        $rmaStatusId = $this->helper->getConfigValue(ConfigPathInterface::RETURNS_STATUS_CONFIG_PATH);

        //Create collection
        $collection = $this->rmaCollectionFactory
            ->create()
            ->addFieldToFilter(
                'main_table.' . RmaInterface::KEY_STATUS_ID,
                ['EQ' => $rmaStatusId]
            );

        $collection
            ->getSelect()
            ->joinLeft(
                ['mri' => 'mst_rma_item'],
                'main_table.rma_id = mri.' . ItemInterface::KEY_RMA_ID,
                []
            )
            ->joinLeft(
                ['soi' => 'sales_order_item'],
                'mri.order_item_id = soi.' . Order\Item::ITEM_ID,
                []
            )
            ->joinLeft(
                ['so' => 'sales_order'],
                'soi.order_id = so.' . Order::ENTITY_ID,
                []
            )
            ->joinLeft(
                ['gps' => $collection->getTable(GelShipmentInterface::TABLE_NAME)],
                'so.' . Order::ENTITY_ID . ' = gps.' . GelShipmentInterface::ORDER_ID,
                [
                    'gel_order_id' => GelShipmentInterface::ORDER_ID,
                    GelShipmentInterface::QUOTE_REFERENCE
                ]
            )
            ->where(
                'gps.'. GelShipmentInterface::IS_RETURN . '=1 AND gps.' . GelShipmentInterface::EXTERNAL_ORDER_ID . ' IS NULL'
            );
        return $collection;
    }

    protected function clearDuplicatesRmas() {
        $duplicates = $this->gelShipmentCollectionFactory
            ->create()
            ->addFieldToFilter(GelShipmentInterface::EXTERNAL_ORDER_ID, ['null' => true])
            ->addFieldToFilter(GelShipmentInterface::IS_RETURN, true);

        $duplicates->getSelect()->group(GelShipmentInterface::QUOTE_REFERENCE)->having('count(*) > 1');

        foreach ($duplicates as $gelShipment) {
            $this->gelShipmentRepository->delete($gelShipment);
        }
    }

    /**
     * Processes the order data and send it to GEL server
     *
     * @param Rma $rma
     * @return array
     */
    protected function processReturn(Rma $rma): array
    {
        $postUrl = $this->helper->getConfigValue(ConfigPathInterface::URL_END_USER_BACKEND_CONFIG_PATH);
        /** @var Order $order */
        $order = $this->orderRepository->get(
            $rma->getData('gel_order_id')
        );
        $order->setData(
            GelShipmentInterface::QUOTE_REFERENCE,
            $rma->getData(GelShipmentInterface::QUOTE_REFERENCE)
        );
        $data = $this->orderProcessor->processConfirm($order, true);
        return $this->gateway->confirmOrder($postUrl, $data);
    }

    /**
     * Retrieve the gel shipment from the quote reference and update with order and external order IDs
     *
     * @param Rma $rma
     * @param array $gelResponse
     * @throws AlreadyExistsException
     * @throws InputException
     * @throws NoSuchEntityException
     * @return void
     */
    protected function updateGelShipment(Rma $rma, array $gelResponse): void
    {
        $gelShipment = $this->gelShipmentRepository->getByQuoteReference(
            $rma->getData(GelShipmentInterface::QUOTE_REFERENCE)
        );
        //Debug
        $this->helper->logDebug('--- [SHIP_RETURNS_SERVICE] Updating GEL shipment ID ' . $gelShipment->getId());
        //Set external order ID
        if (array_key_exists('data', $gelResponse) && array_key_exists('id', $gelResponse['data'])) {
            $gelShipment->setExternalOrderId($gelResponse['data']['id']);
        }
        $this->gelShipmentRepository->save($gelShipment);
        //Debug
        $this->helper->logDebug('--- [SHIP_RETURNS_SERVICE] GEL shipment ID ' . $gelShipment->getId() . ' updated successfully.');
    }

    /**
     * Updates the return status
     *
     * @param Rma $rma
     * @return void
     */
    protected function updateRma(Rma $rma): void
    {
        //Debug
        $this->helper->logDebug('------ [SHIP_RETURNS_SERVICE] Approving Return number '. $rma->getId());
        try {
            $rmaStatus = $this->rmaStatusRepository->getByCode(StatusInterface::APPROVED);
            $rma->setStatusId($rmaStatus->getId());
            $this->rmaRepository->save($rma);
        } catch (NoSuchEntityException $e) {
            $this->helper->log(
                sprintf(
                    '------ [SHIP_RETURNS_SERVICE] Error processing Return %s: the Approved RMA status doesn\'t seems to exist. Created it and try again. Error: %s',
                    $rma->getId(),
                    $e->getMessage()
                ),
                Logger::CRITICAL
            );
        } catch (CouldNotSaveException $e) {
            $this->helper->log(
                sprintf(
                    '------ [SHIP_RETURNS_SERVICE] Error processing Return ID %s -> Error: %s',
                    $rma->getId(),
                    $e->getMessage()
                ),
                Logger::CRITICAL
            );
        }
        //Debug
        $this->helper->logDebug('------ [SHIP_RETURNS_SERVICE] Return number '. $rma->getId() . ' successfully approved.');
    }

    /**
     * Send the error through an email report
     *
     * @param int $orderId
     * @param string $message
     */
    protected function sendErrorReport(int $orderId, string $message): void
    {
        try {
            $this->emailHelper->sendErrorReport($orderId, $message);
        } catch (MailException | LocalizedException $e) {
            //You must be really unlucky to finish here, anyway, log mail error
            $this->helper->log(
                '--- [SHIP_RETURNS_SERVICE] There was an error sending the error report email: ' . $e->getMessage(),
                Logger::ERROR
            );
        }
    }
}
