<?php

namespace Drop\GELProximity\Service;

use Drop\GELProximity\Api\Data\GelShipmentInterface;
use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Api\Service\GelGatewayInterface;
use Drop\GELProximity\Api\Service\Processors\OrderProcessorInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Logger\Logger;
use Drop\GELProximity\Model\ResourceModel\GelShipment\CollectionFactory as GelShipmentCollectionFactory;
use Exception;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;

/**
 * Class CleanService
 * @package Drop\GELProximity\Service
 */
class CleanService extends AbstractService
{
    /**
     * @var GelShipmentCollectionFactory
     */
    protected $gelShipmentCollectionFactory;

    /**
     * CleanService constructor.
     * @param Data $helper
     * @param GelShipmentRepositoryInterface $gelShipmentRepository
     * @param GelGatewayInterface $gateway
     * @param OrderProcessorInterface $orderProcessor
     * @param GelShipmentCollectionFactory $gelShipmentCollectionFactory
     */
    public function __construct(
        Data $helper,
        GelShipmentRepositoryInterface $gelShipmentRepository,
        GelGatewayInterface $gateway,
        OrderProcessorInterface $orderProcessor,
        GelShipmentCollectionFactory $gelShipmentCollectionFactory
    ) {
        parent::__construct($helper, $gelShipmentRepository, $gateway, $orderProcessor);
        $this->gelShipmentCollectionFactory = $gelShipmentCollectionFactory;
    }

    /**
     * Service methods that deletes GEL shipments that hasn't got an order and a quote anymore
     *
     * {@inheritDoc}
     */
    public function run(): int
    {
        if ($this->canBeExecuted()) {
            //Init order collection
            $gelShipmentsCollection = $this->initServiceCollection();
            //Debug
            $this->helper->logDebug(
                sprintf(
                    '[CLEAN_SERVICE] Starting shipments cleaning: found %s shipments to be processed.',
                    $gelShipmentsCollection->getSize()
                )
            );
            /** @var GelShipmentInterface $gelShipment */
            foreach ($gelShipmentsCollection as $gelShipment) {
                //Debug
                $this->helper->logDebug('--- [CLEAN_SERVICE] Deleting GEL shipment with ID ' . $gelShipment->getId());
                try {
                    //Delete the shipment
                    $this->gelShipmentRepository->delete($gelShipment);
                } catch (Exception $e) {
                    $this->helper->log(
                        sprintf(
                            '--- [CLEAN_SERVICE] Error when deleting GEL shipment %s -> Error: %s',
                            $gelShipment->getId(),
                            $e->getMessage()
                        ),
                        Logger::ERROR
                    );
                }
                //Debug
                $this->helper->logDebug('------ [CLEAN_SERVICE] GEL shipment ' . $gelShipment->getId() . ' successfully deleted.');
            }
            return $gelShipmentsCollection->getSize();
        } else {
            //Debug
            $this->helper->logDebug('[CLEAN_SERVICE] Service not enabled: skipping.');
            return 0;
        }
    }

    /**
     * {@inheritDoc}
     */
    protected function initServiceCollection(): AbstractCollection
    {
        return $this->gelShipmentCollectionFactory
            ->create()
            ->addFieldToFilter(GelShipmentInterface::ORDER_ID, ['null' => true])
            ->addFieldToFilter(GelShipmentInterface::QUOTE_ID, ['null' => true]);
    }
}
