<?php

namespace Drop\GELProximity\Api;

use Drop\GELProximity\Api\Data\GelShipmentInterface;
use Drop\GELProximity\Api\Data\GelShipmentSearchResultInterface;
use Exception;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Sales\Api\Data\OrderInterface;

/**
 * Interface GelShipmentRepositoryInterface
 * @package Drop\GELProximity\Api
 */
interface GelShipmentRepositoryInterface
{
    /**
     * Gets a specific GEL shipment by its Entity ID
     *
     * @param int|string $id
     * @return GelShipmentInterface
     * @throws InputException
     * @throws NoSuchEntityException
     */
    public function get(int $id): GelShipmentInterface;

    /**
     * Gets a specific GEL shipment by its external Pickup Point ID
     *
     * @param int $pickupPointId
     * @return GelShipmentInterface
     * @throws InputException
     * @throws NoSuchEntityException
     */
    public function getByPickupPointId(int $pickupPointId): GelShipmentInterface;

    /**
     * Gets a specific GEL shipment by its quote ID mask (used as a reference by GEL API)
     *
     * @param string $quoteReference
     * @return GelShipmentInterface
     * @throws InputException
     * @throws NoSuchEntityException
     */
    public function getByQuoteReference(string $quoteReference): GelShipmentInterface;

    /**
     * Gets a specific GEL shipment by its Quote ID
     *
     * @param int $quoteId
     * @return GelShipmentInterface
     * @throws InputException
     * @throws NoSuchEntityException
     */
    public function getByQuoteId(int $quoteId): GelShipmentInterface;

    /**
     * Gets a specific GEL shipment by its Quote
     *
     * @param CartInterface $quote
     * @return GelShipmentInterface
     * @throws InputException
     * @throws NoSuchEntityException
     */
    public function getByQuote(CartInterface $quote): GelShipmentInterface;

    /**
     * Gets a specific GEL shipment by its Order ID
     *
     * @param int $orderId
     * @param bool $isReturn
     * @return GelShipmentInterface
     * @throws InputException
     * @throws NoSuchEntityException
     */
    public function getByOrderId(int $orderId, bool $isReturn = false): GelShipmentInterface;

    /**
     * Gets a specific GEL shipment by its Order
     *
     * @param OrderInterface $order
     * @param bool $isReturn
     * @return GelShipmentInterface
     * @throws InputException
     * @throws NoSuchEntityException
     */
    public function getByOrder(OrderInterface $order, bool $isReturn = false): GelShipmentInterface;

    /**
     * Saves a GEL shipment entity, updating if there's an entity ID, creating if it's a new one
     *
     * @param GelShipmentInterface $gelShipment
     * @return GelShipmentInterface
     * @throws AlreadyExistsException
     */
    public function save(GelShipmentInterface $gelShipment): GelShipmentInterface;

    /**
     * Deletes a specific GEL shipment by its entity model
     *
     * @param GelShipmentInterface $gelShipment
     * @return bool
     * @throws Exception
     */
    public function delete(GelShipmentInterface $gelShipment): bool;

    /**
     * Deletes a specific GEL shipment by its Entity ID
     *
     * @param int $entityId
     * @return bool
     * @throws InputException
     * @throws NoSuchEntityException
     * @throws Exception
     */
    public function deleteById(int $entityId): bool;

    /**
     * Lists GEL shipments that match a specified search criteria
     *
     * @param SearchCriteriaInterface $searchCriteria
     * @return GelShipmentSearchResultInterface
     */
    public function getList(SearchCriteriaInterface $searchCriteria): GelShipmentSearchResultInterface;
}
