<?php

namespace Drop\GELProximity\Service;

use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Api\OrderStatusInterface;
use Drop\GELProximity\Api\Service\GelGatewayInterface;
use Drop\GELProximity\Api\Service\Processors\OrderProcessorInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Logger\Logger;
use Drop\GELProximity\Model\Carrier\GELProximity;
use Exception;
use Magento\Framework\DB\Select;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;

/**
 * Class SyncOrdersStatusService
 * @package Drop\GELProximity\Service
 */
class SyncOrdersStatusService extends AbstractService
{
    /**
     * @var OrderRepositoryInterface
     */
    protected $orderRepository;

    /**
     * @var CollectionFactory
     */
    protected $orderCollectionFactory;

    /**
     * SyncOrdersStatusService constructor.
     * @param Data $helper
     * @param GelShipmentRepositoryInterface $gelShipmentRepository
     * @param GelGatewayInterface $gateway
     * @param OrderProcessorInterface $orderProcessor
     * @param OrderRepositoryInterface $orderRepository
     * @param CollectionFactory $orderCollectionFactory
     */
    public function __construct(
        Data $helper,
        GelShipmentRepositoryInterface $gelShipmentRepository,
        GelGatewayInterface $gateway,
        OrderProcessorInterface $orderProcessor,
        OrderRepositoryInterface $orderRepository,
        CollectionFactory $orderCollectionFactory
    ) {
        parent::__construct($helper, $gelShipmentRepository, $gateway, $orderProcessor);
        $this->orderRepository = $orderRepository;
        $this->orderCollectionFactory = $orderCollectionFactory;
    }

    /**
     * {@inheritDoc}
     */
    public function run(): int
    {
        if ($this->canBeExecuted()) {
            //Init order collection
            $ordersCollection = $this->initServiceCollection();
            //Debug
            $this->helper->logDebug(
                sprintf(
                    '[SYNC_ORDERS_STATUS_SERVICE] Starting orders status synchronization: found %s orders to be processed.',
                    $ordersCollection->getSize()
                )
            );
            /** @var Order $order */
            foreach ($ordersCollection as $order) {
                try {
                    //Debug
                    $this->helper->logDebug(
                        sprintf(
                            '--- [SYNC_ORDERS_STATUS_SERVICE] Changing status to order %s.',
                            $order->getId()
                        )
                    );

                    //Change order status
                    $this->setOrderInProcess($order);

                    //Debug
                    $this->helper->logDebug(
                        sprintf(
                            '------ [SYNC_ORDERS_STATUS_SERVICE] Status for order %s successfully changed to %s.',
                            $order->getId(),
                            OrderStatusInterface::ORDER_STATUS_PROCESSING_GEL
                        )
                    );
                } catch (Exception $e) {
                    $this->helper->log(
                        sprintf(
                            '--- [SYNC_ORDERS_STATUS_SERVICE] Found an error changing status for order %s. Error: %s',
                            $order->getId(),
                            $e->getMessage()
                        ),
                        Logger::ERROR
                    );
                }
            }
            return $ordersCollection->getSize();
        } else {
            //Debug
            $this->helper->logDebug('[SYNC_ORDERS_STATUS_SERVICE] Service not enabled: skipping.');
            return 0;
        }
    }

    /**
     * {@inheritDoc}
     */
    protected function initServiceCollection(): AbstractCollection
    {
        $collection = $this->orderCollectionFactory
            ->create()
            ->addFieldToSelect('entity_id')
            ->addFieldToFilter('main_table.shipping_method', GELProximity::CARRIER_CODE . '_' . GELProximity::CARRIER_CODE)
            ->addFieldToFilter('main_table.' . Order::STATUS, OrderStatusInterface::ORDER_STATUS_PENDING_GEL)
            ->addFieldToFilter('si.state', Order\Invoice::STATE_PAID)
            ->setOrder('main_table.' . Order::ENTITY_ID, Select::SQL_ASC);
        $collection
            ->getSelect()
            ->joinLeft(
                ['si' => $collection->getTable('sales_invoice')],
                'main_table.entity_id = si.order_id',
                []
            );
        return $collection;
    }

    /**
     * Change the given order status to Processing GEL
     *
     * @param Order $order
     */
    protected function setOrderInProcess(Order $order): void
    {
        $entity = $this->orderRepository->get($order->getId());
        $entity->setState(OrderStatusInterface::ORDER_STATE_PROCESSING_GEL)
            ->setStatus(OrderStatusInterface::ORDER_STATUS_PROCESSING_GEL);
        $this->orderRepository->save($entity);
    }
}
