<?php

namespace Drop\GELProximity\Observer;

use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Api\OrderStatusInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Logger\Logger;
use Drop\GELProximity\Model\Carrier\GELProximity;
use Drop\GELProximity\Model\GelShipmentFactory;
use Magento\Checkout\Model\Session;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Sales\Model\Order;

/**
 * Class SaveShipment
 * @package Drop\GELProximity\Observer
 * @observer
 */
class SaveShipment implements ObserverInterface
{
    /**
     * @var GelShipmentRepositoryInterface
     */
    protected $gelShipmentRepository;

    /**
     * @var GelShipmentFactory
     */
    protected $gelShipmentFactory;

    /**
     * @var Session
     */
    protected $checkoutSession;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * SaveShipment constructor.
     * @param GelShipmentRepositoryInterface $gelShipmentRepository
     * @param GelShipmentFactory $gelShipmentFactory
     * @param Session $checkoutSession
     * @param Data $helper
     */
    public function __construct(
        GelShipmentRepositoryInterface $gelShipmentRepository,
        GelShipmentFactory $gelShipmentFactory,
        Session $checkoutSession,
        Data $helper
    ) {
        $this->gelShipmentRepository = $gelShipmentRepository;
        $this->gelShipmentFactory = $gelShipmentFactory;
        $this->checkoutSession = $checkoutSession;
        $this->helper = $helper;
    }

    /**
     * Observer method used to change the order status before placing it
     * and to save GEL shipment data, stored inside session, to its entity
     *
     * {@inheritDoc}
     */
    public function execute(Observer $observer): void
    {
        if ($this->helper->isModuleEnabled()) {
            /** @var Order $order */
            $order = $observer->getData('order');
            //Debug
            $this->helper->logDebug('[OBSERVER_SAVESHIPMENT] Started changing order status before placing for order ' . $order->getId());

            $shippingMethod = $order->getShippingMethod(true);
            if ($shippingMethod->getData('carrier_code') === GELProximity::CARRIER_CODE) {
                //Change order status to gel shipment only if is not already set as processing
                if (
                    $order->getStatus() !== OrderStatusInterface::ORDER_STATUS_PROCESSING_GEL &&
                    $order->getStatus() !== OrderStatusInterface::ORDER_STATUS_SENT_TO_GEL
                ) {
                    $order->setStatus(OrderStatusInterface::ORDER_STATUS_PENDING_GEL);
                    //Debug
                    $this->helper->logDebug('--- [OBSERVER_SAVESHIPMENT] Changed order status to PENDING GEL for order' . $order->getId());
                }
                //Retrieve shipment data
                $quoteReference = $this->checkoutSession->getData('gel_quote_reference');
                $pickupPointId = $this->checkoutSession->getData('gel_pickup_point_id');
                $shippingCurrency = $this->checkoutSession->getData('gel_shipment_currency');
                $additionalPickupPointData = $this->checkoutSession->getData('gel_additional_pickup_point_data');

                //Debug
                $this->helper->logDebug(sprintf(
                    '--- [OBSERVER_SAVESHIPMENT] Creating entity with pickup point %s, shipping cost %s, masked quote ID %s, order ID %s',
                    $pickupPointId,
                    $order->getShippingAmount(),
                    $quoteReference,
                    $order->getId()
                ));

                //Save gel shipment data
                $gelShipment = $this->gelShipmentFactory
                    ->create()
                    ->setQuoteReference($quoteReference)
                    ->setQuoteId($order->getQuoteId())
                    ->setPickupPointId($pickupPointId)
                    ->setCost($order->getShippingAmount())
                    ->setCurrency($shippingCurrency)
                    ->setAdditionalPickupPointData($additionalPickupPointData);
                try {
                    $this->gelShipmentRepository->save($gelShipment);

                    //Debug
                    $this->helper->logDebug('------ [OBSERVER_SAVESHIPMENT] Entity ' . $gelShipment->getId() . ' created successfully.');

                    //Remove the selected pickup point from the session
                    $this->checkoutSession->setData('gel_shipping_cost', null);
                    $this->checkoutSession->setData('gel_quote_reference', null);
                    $this->checkoutSession->setData('gel_pickup_point_id', null);
                    $this->checkoutSession->setData('gel_shipment_currency', null);
                    $this->checkoutSession->setData('gel_additional_pickup_point_data', null);

                    //Debug
                    $this->helper->logDebug('--------- [OBSERVER_SAVESHIPMENT] Session cleared from GEL data after order place.');
                } catch (AlreadyExistsException $e) {
                    //Debug
                    $this->helper->log(
                        sprintf(
                            '------ [OBSERVER_SAVESHIPMENT] Error creating entity for order %s -> Error: %s',
                            $order->getId(),
                            $e->getMessage()
                        ),
                        Logger::CRITICAL
                    );
                }
            } else {
                //Debug
                $this->helper->logDebug('--- [OBSERVER_SAVESHIPMENT] Order placed without a GEL shipment: skipping.');
            }
        } else {
            //Debug
            $this->helper->logDebug('[OBSERVER_SAVESHIPMENT] Module not enabled: skipping.');
        }
    }
}
