<?php

namespace Drop\GELProximity\Model\Carrier;

use Drop\GELProximity\Api\ConfigPathInterface;
use Drop\GELProximity\Helper\Data;
use Magento\Checkout\Model\Session;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\Quote\Address\RateRequest;
use Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory;
use Magento\Quote\Model\Quote\Address\RateResult\MethodFactory;
use Magento\Quote\Model\Quote\Item\Option;
use Magento\Shipping\Model\Carrier\AbstractCarrier;
use Magento\Shipping\Model\Carrier\CarrierInterface;
use Magento\Shipping\Model\Rate\ResultFactory;
use Psr\Log\LoggerInterface;

/**
 * Class GELProximity
 * @package Drop\GELProximity\Model\Carrier
 */
class GELProximity extends AbstractCarrier implements CarrierInterface
{
    /**
     * @const
     * Carrier code
     */
    public const CARRIER_CODE = 'gelproximity';

    /**
     * @var string
     */
    protected $_code = 'gelproximity';

    /**
     * @var bool
     */
    protected $_isFixed = true;

    /**
     * @var ResultFactory
     */
    protected $rateResultFactory;

    /**
     * @var MethodFactory
     */
    protected $rateMethodFactory;

    /**
     * @var Session
     */
    protected $checkoutSession;

    /**
     * @var CartRepositoryInterface
     */
    protected $quoteRepository;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * GELProximity constructor.
     * @param ScopeConfigInterface $scopeConfig
     * @param ErrorFactory $rateErrorFactory
     * @param LoggerInterface $logger
     * @param ResultFactory $rateResultFactory
     * @param MethodFactory $rateMethodFactory
     * @param Session $checkoutSession
     * @param CartRepositoryInterface $quoteRepository
     * @param Data $helper
     * @param array $data
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        ErrorFactory $rateErrorFactory,
        LoggerInterface $logger,
        ResultFactory $rateResultFactory,
        MethodFactory $rateMethodFactory,
        Session $checkoutSession,
        CartRepositoryInterface $quoteRepository,
        Data $helper,
        array $data = []
    ) {
        $this->rateResultFactory = $rateResultFactory;
        $this->rateMethodFactory = $rateMethodFactory;
        $this->checkoutSession = $checkoutSession;
        $this->quoteRepository = $quoteRepository;
        $this->helper = $helper;
        parent::__construct($scopeConfig, $rateErrorFactory, $logger, $data);
    }

    /**
     * Override of original method to add an extra check for quote items total quantity
     *
     * @return bool
     */
    public function isActive(): bool
    {
        $active = parent::isActive();
        $quoteItemTotQty = $this->getCartTotalQty();
        //Method is active only if total cart qty is less than then number configured
        return $active &&
            $quoteItemTotQty < $this->helper->getConfigValue(ConfigPathInterface::CART_ITEMS_LIMIT_CONFIG_PATH);
    }

    /**
     * {@inheritDoc}
     */
    public function collectRates(RateRequest $request)
    {
        //Debug
        $this->helper->logDebug('[GELPROXIMITY_CARRIER] Starting collecting rates for GEL shipping method.');
        if (!$this->isActive()) {
            //Debug
            $this->helper->logDebug('--- [GELPROXIMITY_CARRIER] Shipping method not enabled: skipping.');
            return false;
        }

        $result = $this->rateResultFactory->create();
        $method = $this->rateMethodFactory->create();

        $method->setCarrier($this->getCarrierCode());
        $method->setCarrierTitle($this->getConfigData('title'));

        $method->setMethod($this->getCarrierCode());
        $method->setMethodTitle($this->getConfigData('name'));

        $shippingCost = $this->getShippingCost();
        //Debug
        $this->helper->logDebug('--- [GELPROXIMITY_CARRIER] Applying a shipping cost of ' . $shippingCost);

        $method->setPrice($shippingCost);
        $method->setCost($shippingCost);

        $result->append($method);

        //Debug
        $this->helper->logDebug('[GELPROXIMITY_CARRIER] Applied the shipping method ' . $this->getConfigData('name') . ' for ' . $shippingCost);
        return $result;
    }

    /**
     * {@inheritDoc}
     */
    public function getAllowedMethods(): array
    {
        return [
            $this->_code => $this->getConfigData('name')
        ];
    }

    /**
     * Returns, if set, the shipping cost from the checkout session
     *
     * @return float
     */
    protected function getShippingCost(): float
    {
        $gelCost = $this->checkoutSession->getData('gel_shipping_cost');
        return $gelCost !== null ? (float)$gelCost : (float)$this->getConfigData('shipping_cost');
    }

    /**
     * Returns the total items qty inside cart
     *
     * @return float
     */
    protected function getCartTotalQty(): float
    {
        $totalQty = 0.0;
        try {
            $quote = $this->quoteRepository->get(
                $this->checkoutSession->getQuoteId()
            );
        } catch (NoSuchEntityException $e) {
            return $totalQty;
        }
        //Calculate total qty of cart
        foreach ($quote->getItems() as $quoteItem) {
            switch ($quoteItem->getProductType()) {
                case 'simple':
                case 'configurable':
                default:
                    $totalQty += (float)$quoteItem->getQty();
                    break;
                case 'bundle':
                    $bundleQty = 0.0;
                    /** @var Option $qtyOption */
                    foreach ($quoteItem->getQtyOptions() as $qtyOption) {
                        $bundleQty += (float)$qtyOption->getValue();
                    }
                    $totalQty += ($bundleQty * $quoteItem->getQty());
                    break;
            }
        }
        return $totalQty;
    }
}
