<?php

namespace Drop\GELProximity\Model\Service\Processors;

use Drop\GELProximity\Api\ConfigPathInterface;
use Drop\GELProximity\Api\Data\GelShipmentInterface;
use Drop\GELProximity\Api\OrderStatusInterface;
use Drop\GELProximity\Api\Service\Processors\OrderProcessorInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Logger\Logger;
use Drop\GELProximity\Model\Carrier\GELProximity;
use Exception;
use Magento\Framework\DB\TransactionFactory;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Shipment\Sender\EmailSender;
use Magento\Shipping\Model\ShipmentNotifier;
use stdClass;

/**
 * Class OrderProcessor
 * @package Drop\GELProximity\Model\Service\Processors
 */
class OrderProcessor implements OrderProcessorInterface
{
    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var \Magento\Sales\Model\Convert\Order
     */
    protected $convertOrder;

    /**
     * @var EmailSender
     */
    protected $shipmentSender;

    /**
     * @var TransactionFactory
     */
    protected $transactionFactory;

    /**
     * @var Order\Shipment\TrackFactory
     */
    protected $trackFactory;

    /**
     * @var ShipmentNotifier
     */
    protected $shipmentNotifier;

    /**
     * @var CartRepositoryInterface
     */
    protected $quoteRepository;

    /**
     * OrderProcessor constructor.
     * @param Data $helper
     * @param \Magento\Sales\Model\Convert\Order $convertOrder
     * @param EmailSender $shipmentSender
     * @param TransactionFactory $transactionFactory
     * @param Order\Shipment\TrackFactory $trackFactory
     * @param ShipmentNotifier $shipmentNotifier
     * @param CartRepositoryInterface $quoteRepository
     */
    public function __construct(
        Data $helper,
        \Magento\Sales\Model\Convert\Order $convertOrder,
        EmailSender $shipmentSender,
        TransactionFactory $transactionFactory,
        Order\Shipment\TrackFactory $trackFactory,
        ShipmentNotifier $shipmentNotifier,
        CartRepositoryInterface $quoteRepository
    ) {
        $this->helper = $helper;
        $this->convertOrder = $convertOrder;
        $this->shipmentSender = $shipmentSender;
        $this->transactionFactory = $transactionFactory;
        $this->trackFactory = $trackFactory;
        $this->shipmentNotifier = $shipmentNotifier;
        $this->quoteRepository = $quoteRepository;
    }

    /**
     * {@inheritDoc}
     */
    public function processConfirm(Order $order, bool $isReturn = false): array
    {
        //Retrieve configuration values
        $apiKey = $this->helper->getConfigValue(ConfigPathInterface::API_KEY_S2S_CONFIG_PATH);
        $merchantCode = $this->helper->getConfigValue(ConfigPathInterface::MERCHANT_CODE_CONFIG_PATH);
        $reference = $order->getData(GelShipmentInterface::QUOTE_REFERENCE);
        //Debug
        $this->helper->logDebug('------ [ORDER_PROCESSOR] Processing data for order ID ' . $order->getId());
        //Create the order item rows
        $orderItemsArray = $this->orderItemsToArray($order->getAllVisibleItems());
        //Create the order data json
        $orderArray = $this->orderToArray($order, $orderItemsArray);
        //Process final body request
        return $this->requestArrayConfirmOrder(
            $apiKey,
            $merchantCode,
            $reference,
            $orderArray,
            $isReturn
        );
    }

    /**
     * {@inheritDoc}
     */
    public function processTracking(Order $order, GelShipmentInterface $gelShipment): array
    {
        //Debug
        $this->helper->logDebug(sprintf(
            '------ [ORDER_PROCESSOR] Processing data for order ID %s and external order ID %s.',
            $order->getId(),
            $gelShipment->getExternalOrderId()
        ));
        $orderObject = new stdClass();
        $orderObject->orderId = $gelShipment->getExternalOrderId();
        return [
            'request' => [
                $orderObject
            ]
        ];
    }

    /**
     * {@inheritDoc}
     */
    public function sentToGEL(Order $order): void
    {
        //Debug
        $this->helper->logDebug('--- [ORDER_PROCESSOR] Changing order ID ' . $order->getId() . ' status to Sent to GEL.');
        $order
            ->setState(OrderStatusInterface::ORDER_STATE_PROCESSING_GEL)
            ->setStatus(OrderStatusInterface::ORDER_STATUS_SENT_TO_GEL);
        $transaction = $this->transactionFactory->create();
        try {
            $transaction
                ->addObject($order)
                ->save();
        } catch (Exception $e) {
            throw new CouldNotSaveException(__($e->getMessage()));
        }
        //Debug
        $this->helper->logDebug(
            sprintf(
                '------ [ORDER_PROCESSOR] Successfully set order ID %s to status Sent to GEL.',
                $order->getId()
            )
        );
    }

    /**
     * {@inheritDoc}
     */
    public function ship(Order $order, int $shipmentStatus): void
    {
        //Debug
        $this->helper->logDebug('--- [ORDER_PROCESSOR] Started shipping the order ID ' . $order->getId());
        //Convert order to shipment
        $shipment = $this->convertOrder
            ->toShipment($order)
            ->setShipmentStatus($shipmentStatus);
        //Process order items
        foreach ($order->getAllItems() as $orderItem) {
            if (!$orderItem->getQtyToShip() || $orderItem->getIsVirtual()) {
                continue;
            }
            $qtyShipped = $orderItem->getQtyToShip();
            try {
                $shipmentItem = $this->convertOrder->itemToShipmentItem($orderItem)->setQty($qtyShipped);
            } catch (LocalizedException $e) {
                $this->helper->log(
                    sprintf(
                        '------ [ORDER_PROCESSOR] Order %s got an error creating the shipment item. Error: %s',
                        $order->getId(),
                        $e->getMessage()
                    ),
                    Logger::WARNING
                );
                continue;
            }
            $shipment->addItem($shipmentItem);
        }
        try {
            $shipment->register();
        } catch (LocalizedException $e) {
            //Will never be thrown this exception
        }
        //Set order in Shipped state
        $shipment->getOrder()
            ->setState(OrderStatusInterface::ORDER_STATUS_SHIPPED_GEL)
            ->setStatus(OrderStatusInterface::ORDER_STATE_SHIPPED_GEL);
        //Save shipment and order entity together
        try {
            $transaction = $this->transactionFactory->create();
            $transaction
                ->addObject($shipment)
                ->addObject($shipment->getOrder())
                ->save();
            //Debug
            $this->helper->logDebug(sprintf(
                '------ [ORDER_PROCESSOR] Successfully created shipment with ID %s from order ID %s',
                $shipment->getId(),
                $order->getId()
            ));
        } catch (Exception $e) {
            throw new CouldNotSaveException(__($e->getMessage()));
        }
        //Notify customer via email
        try {
            $this->shipmentSender->send($order, $shipment);
            //Debug
            $this->helper->logDebug(sprintf(
                '------ [ORDER_PROCESSOR] Successfully notified the customer %s for shipment ID %s from order ID %s',
                $order->getCustomerId() ?: $order->getCustomerName(),
                $shipment->getId(),
                $order->getId()
            ));
        } catch (Exception $e) {
            $this->helper->log(
                sprintf(
                    '------ [ORDER_PROCESSOR] Order %s with shipment %s got an error sending the email notify. Error: %s',
                    $order->getId(),
                    $shipment->getId(),
                    $e->getMessage()
                ),
                Logger::ERROR
            );
        }
    }

    /**
     * {@inheritDoc}
     */
    public function track(Order $order, array $tracking): void
    {
        //Retrieve shipment from order
        /** @var Order\Shipment $shipment */
        $shipment = $order->getShipmentsCollection()->getFirstItem();
        //Debug
        $this->helper->logDebug(sprintf(
            '------ [ORDER_PROCESSOR] Creating tracking entity for shipment %s and order %s',
            $shipment->getId(),
            $order->getId()
        ));
        //Init tracking entity
        $track = $this->trackFactory
            ->create()
            ->setShipment($shipment)
            ->setWeight($order->getWeight())
            ->setQty($order->getTotalQtyOrdered())
            ->setOrderId($order->getId())
            ->setTrackNumber($tracking['number'])
            ->setDescription($tracking['statusDescription'])
            ->setTitle($this->helper->getConfigValue('carriers/gelproximity/title'))
            ->setCarrierCode(GELProximity::CARRIER_CODE);
        try {
            //Save shipment and tracking entities
            $transaction = $this->transactionFactory->create();
            $transaction
                ->addObject($shipment)
                ->addObject($track)
                ->save();
            //Debug
            $this->helper->logDebug(sprintf(
                '--------- [ORDER_PROCESSOR] Successfully created tracking with ID %s from order %s',
                $track->getId(),
                $order->getId()
            ));
        } catch (Exception $e) {
            throw new CouldNotSaveException(__($e->getMessage()));
        }
        //Notify the customer of the shipment tracking creation
        try {
            $this->shipmentNotifier->notify($shipment);
            //Debug
            $this->helper->logDebug(sprintf(
                '--------- [ORDER_PROCESSOR] Successfully notified the customer %s for tracking %s from order %s',
                $order->getCustomerId() ?: $order->getCustomerName(),
                $track->getId(),
                $order->getId()
            ));
        } catch (MailException $e) {
            $this->helper->log(
                sprintf(
                    '--------- [ORDER_PROCESSOR] Order %s with tracking %s got an error sending the email notify. Error: %s',
                    $order->getId(),
                    $track->getId(),
                    $e->getMessage()
                ),
                Logger::ERROR
            );
        }
    }

    /**
     * {@inheritDoc}
     */
    public function complete(Order $order): void
    {
        //Debug
        $this->helper->logDebug('--- [ORDER_PROCESSOR] Completing the order ' . $order->getId());
        $order
            ->setState(Order::STATE_COMPLETE)
            ->setStatus($order->getConfig()->getStateDefaultStatus(Order::STATE_COMPLETE));
        $transaction = $this->transactionFactory->create();
        try {
            $transaction
                ->addObject($order)
                ->save();
        } catch (Exception $e) {
            throw new CouldNotSaveException(__($e->getMessage()));
        }
        //Debug
        $this->helper->logDebug(sprintf(
            '------ [ORDER_PROCESSOR] Completed the order %s',
            $order->getId()
        ));
    }

    /**
     * Returns the array containing the main request body
     *
     * @param string $apiKey
     * @param string $merchantCode
     * @param string $reference
     * @param array $orderArray
     * @param bool $isReturn
     * @return array
     */
    protected function requestArrayConfirmOrder(
        string $apiKey,
        string $merchantCode,
        string $reference,
        array $orderArray,
        bool $isReturn
    ): array {

        // invio dimensioni pacco
        if($isReturn) {
            $orderArray = array_merge($orderArray,[
                'height' => 41,
                'width' => 57,
                'depth' => 15,
                'volume' => 35055
            ]);
        }

        return [
            'apiKey' => $apiKey,
            'merchantCode' => $merchantCode,
            'reference' => $reference,
            'orderDataJson' => $this->helper->serializeArray($orderArray),
            'isReturnOrder' => $isReturn ? 'true' : 'false'
        ];
    }

    /**
     * Returns the array with the order and order items main data
     *
     * @param Order $order
     * @param array $orderItemsArray
     * @return array
     */
    protected function orderToArray(Order $order, array $orderItemsArray): array
    {
        try {
            //Retrieve shipping address from quote
            $quote = $this->quoteRepository->get($order->getQuoteId());
            $shippingAddress = $quote->getShippingAddress();
        } catch (NoSuchEntityException $e) {
            $this->helper->log(
                sprintf(
                    '[ORDER_PROCESSOR] Found an error processing order %s. Error: %s',
                    $order->getId(),
                    $e->getMessage()
                ),
                Logger::ERROR
            );
            //Fallback shipping address to the order
            $shippingAddress = $order->getShippingAddress();
        }
        $customerName = sprintf('%s %s', $shippingAddress->getFirstname(), $shippingAddress->getLastname());
        return [
            'orderNumber' => $order->getIncrementId(),
            'orderNumberRif' => $order->getIncrementId(),
            'reference' => $order->getIncrementId(),
            'orderDate' => date('Ymd', strtotime($order->getCreatedAt())),
            'description' => sprintf(
                'Ordine %s - %s',
                $order->getIncrementId(),
                date('d.m.Y', strtotime($order->getCreatedAt()))
            ),
            'customer' => $customerName,
            'customerEmail' => $order->getCustomerEmail(),
            'customerPhone' => $shippingAddress->getTelephone(),
            'codePickingPoint' => $order->getData(GelShipmentInterface::PICKUP_POINT_ID),
            'grossWeight' => $order->getWeight(),
            'netWeight' => $order->getWeight(),
            'goodsValue' => $order->getSubtotalInvoiced(),
            'currencyGoodsValue' => $order->getOrderCurrencyCode(),
            'piecesCount' => (int)$order->getTotalQtyOrdered(),
            'rows' => $orderItemsArray
        ];
    }

    /**
     * Returns an array containing all the visible ordered items
     *
     * @param array $orderItems
     * @return array
     */
    protected function orderItemsToArray(array $orderItems): array
    {
        $rows = [];
        foreach ($orderItems as $orderItem) {
            $rows[] = [
                'description' => $orderItem->getDescription() ?: sprintf(
                    '%s - %s - type %s',
                    $orderItem->getSku(),
                    $orderItem->getName(),
                    $orderItem->getProductType()
                ),
                'productCode' => $orderItem->getSku(),
                'typeProduct' => $orderItem->getProductType(),
                'grossWeight' => $orderItem->getRowWeight(),
                'netWeight' => $orderItem->getRowWeight()
            ];
        }
        return $rows;
    }
}
