<?php

namespace Drop\GELProximity\Model\Service;

use Drop\GELProximity\Api\Service\GelServiceInterface;
use Drop\GELProximity\Api\Service\Responses\CurrentPickupPointInterface;
use Drop\GELProximity\Api\Service\Responses\DefaultResponseInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Model\Service\Responses\ResponseObjectFactory;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Customer\Model\Session as CustomerSession;

/**
 * Class GelService
 * @package Drop\GELProximity\Model\Service
 */
class GelService implements GelServiceInterface
{
    /**
     * @var CheckoutSession
     */
    protected $checkoutSession;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var ResponseObjectFactory
     */
    protected $responseObjectFactory;

    /**
     * @var CustomerSession
     */
    protected $customerSession;

    /**
     * GelService constructor.
     * @param CheckoutSession $checkoutSession
     * @param Data $helper
     * @param ResponseObjectFactory $responseObjectFactory
     * @param CustomerSession $customerSession
     */
    public function __construct(
        CheckoutSession $checkoutSession,
        Data $helper,
        ResponseObjectFactory $responseObjectFactory,
        CustomerSession $customerSession
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->helper = $helper;
        $this->responseObjectFactory = $responseObjectFactory;
        $this->customerSession = $customerSession;
    }

    /**
     * {@inheritDoc}
     */
    public function removePickupPoint(): DefaultResponseInterface
    {
        /** @var DefaultResponseInterface $response */
        $response = $this->responseObjectFactory->create(ResponseObjectFactory::DEFAULT_RESPONSE);
        if ($this->helper->isModuleEnabled()) {
            $quoteId = $this->checkoutSession->getQuoteId();
            //Debug
            $this->helper->logDebug('[GELSERVICE_REMOVEPICKUP] Removing pickup point data from quote ID ' . $quoteId . ' session');
            //Remove the selected pickup point from the session
            $this->checkoutSession->setData('gel_shipping_cost', null);
            $this->checkoutSession->setData('gel_quote_reference', null);
            $this->checkoutSession->setData('gel_pickup_point_id', null);
            $this->checkoutSession->setData('gel_shipment_currency', null);
            $this->checkoutSession->setData('gel_additional_pickup_point_data', null);
            //Debug
            $this->helper->logDebug('--- [GELSERVICE_REMOVEPICKUP] Removed pickup point data from session belonging to quote ID ' . $quoteId);
            return $response
                ->setSuccess(true)
                ->setMessage('Pickup point successfully removed.');
        }
        //Debug
        $this->helper->logDebug('[GELSERVICE_REMOVEPICKUP] Module not enabled: skipping.');
        return $response
            ->setSuccess(false)
            ->setMessage('There was a problem processing your request.');
    }

    /**
     * {@inheritDoc}
     */
    public function addPickupPoint(array $data): DefaultResponseInterface
    {
        /** @var DefaultResponseInterface $response */
        $response = $this->responseObjectFactory->create(ResponseObjectFactory::DEFAULT_RESPONSE);
        if ($this->helper->isModuleEnabled()) {
            //Debug
            $this->helper->logDebug('[GELSERVICE_ADDPICKUP] Response from GEL received: ' . $this->helper->serializeArray($data));
            $quoteReference = $data['reference'];
            if ($quoteReference) {
                $additionalPickupPointData = $this->getAdditionalPickupPointData($data);
                //Save GEL data inside checkout session
                $this->checkoutSession->setData('gel_shipping_cost', $data['cost']);
                $this->checkoutSession->setData('gel_quote_reference', $quoteReference);
                $this->checkoutSession->setData('gel_pickup_point_id', $data['pickupPointId']);
                $this->checkoutSession->setData('gel_shipment_currency', $data['currencyCost']);
                $this->checkoutSession->setData('gel_additional_pickup_point_data', $additionalPickupPointData);
                //Debug
                $this->helper->logDebug('--- [GELSERVICE_ADDPICKUP] Added pickup point data from session belonging to quote ID ' . $this->checkoutSession->getQuoteId());
                return $response
                    ->setSuccess(true)
                    ->setMessage('Pickup point successfully added.');
            } else {
                $this->helper->logDebug('--- [GELSERVICE_ADDPICKUP] No quote exist in response. Probably someone trying to call this without permissions.');
            }
        } else {
            //Debug
            $this->helper->logDebug('[GELSERVICE_ADDPICKUP] Module not enabled: skipping.');
        }
        return $response
            ->setSuccess(false)
            ->setMessage('There was a problem processing your request.');
    }

    /**
     * {@inheritDoc}
     */
    public function getCurrentPickupPoint(): CurrentPickupPointInterface
    {
        /** @var CurrentPickupPointInterface $response */
        $response = $this->responseObjectFactory->create(ResponseObjectFactory::CURRENT_PICKUP_POINT_RESPONSE);
        if ($this->helper->isModuleEnabled()) {
            //Debug
            $this->helper->logDebug('[GELSERVICE_GETCURRENTPICKUPDATA] Retrieving data for quote ID session ' . $this->checkoutSession->getQuoteId());
            $additionalPickupPointData = $this->checkoutSession->getData('gel_additional_pickup_point_data');
            if ($additionalPickupPointData !== null) {
                //Debug
                $this->helper->logDebug('[GELSERVICE_GETCURRENTPICKUPDATA] Data for quote ID ' . $this->checkoutSession->getQuoteId() . ': ' . $additionalPickupPointData);
                $response
                    ->setPickupPointData($additionalPickupPointData)
                    ->setSuccess(true)
                    ->setMessage('Pickup point data successfully retrieved.');
            } else {
                $response
                    ->setSuccess(false)
                    ->setMessage('There was a problem retrieving pickup point data.');
            }
        } else {
            //Debug
            $this->helper->logDebug('[GELSERVICE_ADDPICKUP] Module not enabled: skipping.');
            $response
                ->setSuccess(false)
                ->setMessage('There was a problem processing your request.');
        }
        return $response;
    }

    /**
     * {@inheritDoc}
     */
    public function addRMAPickupPoint(
        array $data,
        ?int $orderId
    ): DefaultResponseInterface {
        $response = $this->responseObjectFactory->create(ResponseObjectFactory::DEFAULT_RESPONSE);
        if ($this->helper->isModuleEnabled()) {
            //Debug
            $this->helper->logDebug('[GELSERVICE_ADDRMAPICKUPPOINT] Adding in session for order ID ' . $orderId . ' -> Data: ' . $this->helper->serializeArray($data));
            $this->customerSession->setData('gel_rma_pickup_point_data', $data);
            $this->customerSession->setData('gel_rma_order_id', $orderId);
            $response
                ->setSuccess(true)
                ->setMessage('Pickup point successfully added.');
        } else {
            //Debug
            $this->helper->logDebug('[GELSERVICE_ADDRMAPICKUPPOINT] Module not enabled: skipping.');
            $response
                ->setSuccess(false)
                ->setMessage('There was a problem processing your request.');
        }
        return $response;
    }

    /**
     * {@inheritDoc}
     */
    public function removeRMAPickupPoint(): DefaultResponseInterface
    {
        $response = $this->responseObjectFactory->create(ResponseObjectFactory::DEFAULT_RESPONSE);
        if ($this->helper->isModuleEnabled()) {
            $orderId = $this->customerSession->getData('gel_rma_order_id');
            //Debug
            $this->helper->logDebug('[GELSERVICE_REMOVERMAPICKUPPOINT] Removing from session pickup point selection of order ID ' . $orderId);
            $this->customerSession->setData('gel_rma_pickup_point_data', null);
            $this->customerSession->setData('gel_rma_order_id', null);
            $response
                ->setSuccess(true)
                ->setMessage('Pickup point successfully removed.');
        } else {
            //Debug
            $this->helper->logDebug('[GELSERVICE_REMOVERMAPICKUPPOINT] Module not enabled: skipping.');
            $response
                ->setSuccess(false)
                ->setMessage('There was a problem processing your request.');
        }
        return $response;
    }

    /**
     * Retrieves all the additional pickup point data from the GEL callback response and serialize it into a JSON
     *
     * @param string[] $data
     * @return string
     */
    protected function getAdditionalPickupPointData(array $data): string
    {
        $additionalData = [
            'code' => $data['code'],
            'description' => $data['description'],
            'address' => $data['address'],
            'city' => $data['city'],
            'department' => $data['department'],
            'zipCode' => $data['zipCode'],
            'country' => $data['country'],
        ];
        return $this->helper->serializeArray($additionalData);
    }
}
