<?php

namespace Drop\GELProximity\Helper;

use Drop\GELProximity\Api\ConfigPathInterface;
use Drop\GELProximity\Api\Helper\EmailInterface;
use Magento\Framework\App\Area;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Framework\Translate\Inline\StateInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\Store;

/**
 * Class Email
 * @package Drop\GELProximity\Helper
 */
class Email extends AbstractHelper implements EmailInterface
{
    /**
     * @var StateInterface
     */
    protected $inlineTranslation;

    /**
     * @var TransportBuilder
     */
    protected $transportBuilder;

    /**
     * Email constructor.
     * @param Context $context
     * @param StateInterface $inlineTranslation
     * @param TransportBuilder $transportBuilder
     */
    public function __construct(
        Context $context,
        StateInterface $inlineTranslation,
        TransportBuilder $transportBuilder
    ) {
        parent::__construct($context);
        $this->inlineTranslation = $inlineTranslation;
        $this->transportBuilder = $transportBuilder;
    }

    /**
     * {@inheritDoc}
     */
    public function sendErrorReport(int $orderId, string $errorMessage): void
    {
        $enabled = $this->scopeConfig->isSetFlag(
            ConfigPathInterface::ERROR_REPORT_ENABLED_CONFIG_PATH,
            ScopeInterface::SCOPE_WEBSITE
        );
        if ($enabled) {
            //Init mail configuration
            $templateVars = [
                'orderId' => $orderId,
                'errorMessage' => $errorMessage
            ];
            $sender = $this->getEmailGeneralContact();
            $recipients = explode(
                ',',
                $this->scopeConfig->getValue(
                    ConfigPathInterface::ERROR_REPORT_SEND_TO_CONFIG_PATH,
                    ScopeInterface::SCOPE_WEBSITE
                )
            );
            //Send email
            $this->send(
                'email_cron_error_report',
                $templateVars,
                $sender,
                $recipients
            );
        }
    }

    /**
     * {@inheritDoc}
     */
    public function send(
        string $templateId,
        array $templateVars,
        array $sender,
        array $recipients,
        string $area = Area::AREA_ADMINHTML
    ): void {
        //Suspend translations
        $this->inlineTranslation->suspend();
        //Configure template options
        $this->configureTemplate($templateId, $area, $templateVars);
        //Add email sender
        $this->setSender($sender);
        //Add recipient(s)
        $this->setRecipients($recipients);
        //Get message transport and send email
        $transport = $this->transportBuilder->getTransport();
        $transport->sendMessage();
        //Resume translation
        $this->inlineTranslation->resume();
    }

    /**
     * Sets the template options, variables and identifier for the transport
     *
     * @param string $templateId
     * @param string $area
     * @param array $templateVars
     * @return TransportBuilder
     */
    protected function configureTemplate(
        string $templateId,
        string $area,
        array $templateVars
    ): TransportBuilder {
        return $this->transportBuilder
            ->setTemplateIdentifier($templateId)
            ->setTemplateOptions([
                'area' => $area,
                'store' => Store::DEFAULT_STORE_ID
            ])
            ->setTemplateVars($templateVars);
    }

    /**
     * Sets the email sender for the transport
     *
     * @param array $sender
     * @param int|null $scopeId
     * @return TransportBuilder
     */
    protected function setSender(array $sender, int $scopeId = null): TransportBuilder
    {
        return $this->transportBuilder->setFromByScope($sender, $scopeId);
    }

    /**
     * Add recipients to the transport
     *
     * @param array $recipients
     * @return TransportBuilder
     */
    protected function setRecipients(array $recipients): TransportBuilder
    {
        foreach ($recipients as $recipient) {
            $this->transportBuilder->addTo(trim($recipient));
        }
        return $this->transportBuilder;
    }

    /**
     * Returns an array containing name and email of general contact address
     *
     * @return array
     */
    protected function getEmailGeneralContact(): array
    {
        return [
            'name' => $this->scopeConfig->getValue('trans_email/ident_general/name', ScopeInterface::SCOPE_WEBSITE),
            'email' => $this->scopeConfig->getValue('trans_email/ident_general/email', ScopeInterface::SCOPE_WEBSITE)
        ];
    }
}
