define([
    'jquery',
    'ko',
    'Magento_Ui/js/form/element/abstract',
    'Magento_Ui/js/model/messageList',
    'Magento_Checkout/js/model/full-screen-loader',
    'Magento_Checkout/js/model/quote',
    'Magento_Checkout/js/model/shipping-rate-registry',
    'GelLib',
    'mage/translate',
    'mage/storage'
], function ($, ko, Component, messageList, fullScreenLoader, mainQuote, rateRegistry,  GelLib, $t, storage) {
    'use strict';

    return Component.extend({
        defaults: {
            componentText: '',
            apiKey: '',
            merchantCode: '',
            reference: '',
            redirectMode: '',
            urlRedirectOk: '',
            urlRedirectCancel: '',
            locale: '',
            urlEndUser: '',
            pickupPointId: '',
            pickupPointData: ''
        },
        pickupSelected: '',
        pickupPointFormatted: '',
        gelSDK: {},

        /**
         * @constructor
         */
        initialize: function () {
            this._super()
                ._initGel();
            this.pickupSelected = ko.observable(this.pickupPointId !== null);
            this.pickupPointFormatted = ko.observable(this.pickupSelection());
            //Bind pickup point validation custom event to component private function
            $(document).on('pickupPointId:validation', this._triggerValidation.bind(this));
        },

        /**
         * Returns the text of the pickup point button
         *
         * @returns {string}
         */
        getComponentText: function () {
            return this.componentText;
        },

        /**
         * Action event for the button which will open the gel-end-user
         *
         * @event
         */
        selectPickupPoint: function () {
            this.gelSDK.createUIModal({
                callbackOk: (response) => this.addPickupPoint(response),
                callbackKo: () => {}
            });
        },

        /**
         * Callback action when the pickup point get selected
         *
         * @param callbackData
         * @returns {Promise<void>}
         */
        addPickupPoint: async function (callbackData) {
            fullScreenLoader.startLoader();
            let serviceUrl = 'rest/V1/gel/pickupPoint/select';
            storage.post(
                serviceUrl,
                JSON.stringify({data: callbackData})
            ).then(response => {
                if (response.success) {
                    //Add pickup point selection
                    this.pickupSelected(true);
                    this.pickupPointId = callbackData.pickupPointId;
                    $('#pickup_point_id').val(callbackData.pickupPointId);
                    this.pickupPointData = JSON.stringify(callbackData);
                    this.pickupPointFormatted(this.pickupSelection());
                    //Refresh shipping rates
                    this.refreshShippingMethods();
                    //Trigger custom event to format the address for the sidebar KO component
                    $(document.body).trigger(
                        'pickupPointId:selection',
                        [callbackData]
                    );
                    //Add success message
                    messageList.addSuccessMessage({message: response.message})
                } else {
                    //Add error message
                    messageList.addErrorMessage({message: response.message})
                }
                fullScreenLoader.stopLoader();
            })
        },

        /**
         * Action event which will remove the selected pickup point
         *
         * @event
         */
        removePickupPoint: async function () {
            fullScreenLoader.startLoader();
            //Webapi call to remove pickup point from session
            let serviceUrl = 'rest/V1/gel/pickupPoint/remove',
                payload = {};
            storage.post(
                serviceUrl,
                JSON.stringify(payload)
            ).then(response => {
                if (response.success) {
                    //Remove pickup point selection
                    this.pickupSelected(false);
                    $('#pickup_point_id').val('');
                    //Refresh shipping rates
                    this.refreshShippingMethods();
                    //Add success message
                    messageList.addSuccessMessage({message: response.message})
                } else {
                    messageList.addErrorMessage({message: response.message})
                }
                fullScreenLoader.stopLoader();
            });
        },

        /**
         * Returns the selected pickup point
         *
         * @returns {string}
         */
        pickupSelection: function () {
            let pickupPointData = JSON.parse(this.pickupPointData);
            if (pickupPointData !== null) {
                return '%1, %2 (%3), %4, %5'
                    .replace('%1', pickupPointData.address)
                    .replace('%2', pickupPointData.city)
                    .replace('%3', pickupPointData.department)
                    .replace('%4', pickupPointData.zipCode)
                    .replace('%5', pickupPointData.country);
            }
            return '';
        },

        /**
         * Refresh the shipping method component to re-collect the shipping rates
         */
        refreshShippingMethods: function () {
            let address = mainQuote.shippingAddress();
            rateRegistry.set(address.getKey(), null);
            rateRegistry.set(address.getCacheKey(), null);
            mainQuote.shippingAddress(address);
        },

        /**
         * Trigger the validation handler for this component with a custom event
         *
         * @param {Event} event
         * @param {int|string} value
         * @param {string} shippingMethod
         * @private
         */
        _triggerValidation: function (event, value, shippingMethod) {
            if (
                shippingMethod === 'gelproximity' &&
                (!value || value === '')
            ) {
                let message = $t('You must select a pickup point before continuing.');
                this.error(message);
                this.bubble('error', message);
                return;
            }
            this.error(false);
        },

        /**
         * Init GelSDK object
         *
         * @private
         */
        _initGel: function () {
            //Init the GelSDK object with requested data
            const postData = this._initRequestObject();
            this.gelSDK = new GelLib.GelSDK(postData);
        },

        /**
         * Creates an object containing the POST object that will be sent to GEL API
         *
         * @returns {{apiKey, merchantCode, reference, redirectMode, urlRedirectOk, urlRedirectCancel, locale, urlEndUser}}
         * @private
         */
        _initRequestObject: function () {
            return {
                apiKey: this.apiKey,
                merchantCode: this.merchantCode,
                reference: this.reference,
                redirectMode: this.redirectMode,
                urlRedirectOk: this.urlRedirectOk,
                urlRedirectCancel: this.urlRedirectCancel,
                locale: this.locale,
                urlEndUser: this.urlEndUser
            };
        }
    })
})
