<?php

namespace Drop\GELProximity\Helper;

use Drop\GELProximity\Api\ConfigPathInterface;
use Drop\GELProximity\Logger\Logger;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Locale\Resolver;
use Magento\Framework\Serialize\Serializer\Json;
use Drop\GELProximity\Api\Helper\MaskedQuoteIdToQuoteIdInterface;
use Drop\GELProximity\Api\Helper\QuoteIdToMaskedQuoteIdInterface;
use Magento\Store\Model\ScopeInterface;

/**
 * Class Data
 * @package Drop\GELProximity\Helper
 */
class Data extends AbstractHelper implements ConfigPathInterface
{
    /**
     * @var Resolver
     */
    protected $localeResolver;

    /**
     * @var QuoteIdToMaskedQuoteIdInterface
     */
    protected $quoteIdMasker;

    /**
     * @var MaskedQuoteIdToQuoteIdInterface
     */
    protected $quoteIdUnmasker;

    /**
     * @var Json
     */
    protected $jsonSerializer;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * Data constructor.
     * @param Context $context
     * @param Resolver $localeResolver
     * @param QuoteIdToMaskedQuoteIdInterface $quoteIdMasker
     * @param MaskedQuoteIdToQuoteIdInterface $quoteIdUnmasker
     * @param Json $jsonSerializer
     * @param Logger $logger
     */
    public function __construct(
        Context $context,
        Resolver $localeResolver,
        QuoteIdToMaskedQuoteIdInterface $quoteIdMasker,
        MaskedQuoteIdToQuoteIdInterface $quoteIdUnmasker,
        Json $jsonSerializer,
        Logger $logger
    ) {
        $this->localeResolver = $localeResolver;
        $this->quoteIdMasker = $quoteIdMasker;
        $this->quoteIdUnmasker = $quoteIdUnmasker;
        $this->jsonSerializer = $jsonSerializer;
        $this->logger = $logger;
        parent::__construct($context);
    }

    /**
     * Returns if the module is enabled
     *
     * @return bool
     */
    public function isModuleEnabled(): bool
    {
        return $this->getConfigValue(
            self::ENABLED_CONFIG_PATH
        );
    }

    /**
     * Returns if the debug mode is enabled
     *
     * @return bool
     */
    public function isDebugModeEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(
            self::DEBUG_MODE_CONFIG_PATH,
            ScopeInterface::SCOPE_WEBSITE
        );
    }

    /**
     * Returns the environment that will be used for the authentication
     *
     * @return string
     */
    public function getEnvironment(): string
    {
        return $this->scopeConfig->getValue(
            self::ENVIRONMENT_CONFIG_PATH,
            ScopeInterface::SCOPE_WEBSITE
        );
    }

    /**
     * Returns the config value based on the selected environment
     *
     * @param string $path
     * @return string
     */
    public function getConfigValue(string $path): string
    {
        return $this->scopeConfig->getValue(
            sprintf(
                $path,
                $this->getEnvironment()
            ),
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * Log message only if debug mode is enabled
     *
     * @param string $message
     * @return bool
     */
    public function logDebug(string $message): bool
    {
        if ($this->isDebugModeEnabled()) {
            //Using info instead of debug method because not working properly
            return $this->logger->info($message);
        }
        return false;
    }

    /**
     * Log message based on the level
     *
     * @param string $message
     * @param int $level
     * @return bool
     */
    public function log(string $message, int $level): bool
    {
        return $this->logger->log($level, $message);
    }

    /**
     * Returns the locale of the current store or the default one
     *
     * @return string
     */
    public function getLocale(): string
    {
        return $this->localeResolver->getLocale();
    }

    /**
     * Returns the masked version of the given Quote ID
     *
     * @param int $quoteId
     * @return string
     * @throws NoSuchEntityException
     */
    public function maskQuoteId(int $quoteId): string
    {
        return $this->quoteIdMasker->execute($quoteId);
    }

    /**
     * Returns the Quote ID from the given Quote ID Mask
     *
     * @param string $maskedQuoteId
     * @return int
     * @throws NoSuchEntityException
     */
    public function unmaskQuoteId(string $maskedQuoteId): int
    {
        return $this->quoteIdUnmasker->execute($maskedQuoteId);
    }

    /**
     * Serialize an array into a string
     *
     * @param array $data
     * @return string
     */
    public function serializeArray(array $data): string
    {
        return $this->jsonSerializer->serialize($data);
    }

    /**
     * Unserialize a string to an associative array
     *
     * @param string $data
     * @return array
     */
    public function unserializeToArray(string $data): array
    {
        return $this->jsonSerializer->unserialize($data);
    }

    /**
     * Creates the container and component for GEL Proximity shipping method
     *
     * @param array $jsLayout
     * @param string $quoteReference
     * @param string|null $pickupPointId
     * @param string|null $additionalPickupPointData
     */
    public function addGELButtonToCheckout(
        array &$jsLayout,
        string $quoteReference,
        ?string $pickupPointId,
        ?string $additionalPickupPointData
    ): void {
        //Create component layout
        $component = [
            'component' => 'uiComponent',
            'displayArea' => 'shippingAdditional',
            'children' => [
                'gel_proximity' => [
                    'component' => 'Drop_GELProximity/js/view/gel-proximity-block',
                    'displayArea' => 'gel-proximity-block',
                    'deps' => 'checkoutProvider',
                    'dataScopePrefix' => 'gel_proximity',
                    'children' => [
                        'form-fields' => [
                            'component' => 'uiComponent',
                            'displayArea' => 'gel-proximity-block',
                            'children' => [
                                'messages' => [
                                    'component' => 'Magento_Ui/js/view/messages',
                                    'displayArea' => 'messages'
                                ],
                                'gel-proximity-button' => [
                                    'component' => 'Drop_GELProximity/js/view/components/gel-proximity-button',
                                    'config' => [
                                        'template' => 'ui/form/field',
                                        'elementTmpl' => 'Drop_GELProximity/components/gel-proximity-button',
                                    ],
                                    'provider' => 'checkoutProvider',
                                    'sortOrder' => 950,
                                    'visible' => true,
                                    'pickupPointId' => $pickupPointId,
                                    'pickupPointData' => $additionalPickupPointData,
                                    'componentText' => $this->getConfigValue(self::BUTTON_TEXT_CONFIG_PATH),
                                    'apiKey' => $this->getConfigValue(self::API_KEY_CONFIG_PATH),
                                    'merchantCode' => $this->getConfigValue(self::MERCHANT_CODE_CONFIG_PATH),
                                    'reference' => $quoteReference,
                                    'redirectMode' => $this->getConfigValue(self::REDIRECT_MODE_CONFIG_PATH),
                                    'urlRedirectOk' => $this->_urlBuilder->getUrl(
                                        $this->getConfigValue(self::PATH_REDIRECT_OK_CONFIG_PATH)
                                    ),
                                    'urlRedirectCancel' => $this->_urlBuilder->getUrl(
                                        $this->getConfigValue(self::PATH_REDIRECT_CANCEL_CONFIG_PATH)
                                    ),
                                    'locale' => $this->getLocale(),
                                    'urlEndUser' => $this->getConfigValue(self::URL_END_USER_CLIENT_CONFIG_PATH)
                                ]
                            ],
                        ],
                    ]
                ]
            ]
        ];

        $jsLayout['components']['checkout']['children']['steps']['children']['shipping-step']
        ['children']['shippingAddress']['children']['shippingAdditional'] = $component;
    }

    /**
     * Adds selected pickup point information inside checkout sidebar
     *
     * @param array $jsLayout
     * @param string|null $additionalPickupPointData
     */
    public function addGELInfoToCheckout(
        array &$jsLayout,
        ?string $additionalPickupPointData
    ): void {
        $component = [
            'component' => 'Drop_GELProximity/js/view/pickup-point',
            'config' => [
                'template' => 'Drop_GELProximity/view/pickup-point'
            ],
            'displayArea' => 'shipping-information',
            'pickupPointData' => $additionalPickupPointData
        ];
        //Add pickup point information review to sidebar in checkout
        $jsLayout['components']['checkout']['children']['sidebar']['children']['pickup-point'] = $component;
    }
}
