<?php

namespace Drop\GELProximity\Model\Service;

use Drop\GELProximity\Api\ConfigPathInterface;
use Drop\GELProximity\Helper\Data;
use Magento\Framework\Exception\AuthenticationException;
use Zend\Http\Client;
use Drop\GELProximity\Api\Service\GelGatewayInterface;
use Zend\Http\Header\Accept;
use Zend\Http\Header\Authorization;
use Zend\Http\Header\ContentType;
use Zend\Http\Headers;
use Zend\Http\Request;
use Zend\Http\Response;

/**
 * Class GelGateway
 * @package Drop\GELProximity\Model\Service
 */
class GelGateway implements GelGatewayInterface
{
    /**
     * @var Client
     */
    protected $client;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var string
     */
    protected $accessToken = '';

    /**
     * GelGateway constructor.
     * @param Client $client
     * @param Data $helper
     */
    public function __construct(
        Client $client,
        Data $helper
    ) {
        $this->client = $client;
        $this->helper = $helper;
    }

    /**
     * {@inheritDoc}
     * @throws AuthenticationException
     */
    public function authenticate(): void
    {
        //Gets config data
        $data = $this->helper->serializeArray(
            [
                'apiKey' => $this->helper->getConfigValue(ConfigPathInterface::API_KEY_S2S_CONFIG_PATH),
                'merchantCode' => $this->helper->getConfigValue(ConfigPathInterface::MERCHANT_CODE_CONFIG_PATH)
            ]
        );
        $url = $this->helper->getConfigValue(ConfigPathInterface::URL_GEL_API_SERVER_CONFIG_PATH);
        //Make API call to auth on GEL server
        $rawResponse = $this->post(
            $url,
            'merchant/autenticate',
            $data,
            'application/json'
        );
        $authArray = $this->helper->unserializeToArray($rawResponse->getBody());
        //Save authentication token
        if (array_key_exists('success', $authArray) && $authArray['success']) {
            $this->accessToken = sprintf(
                '%s %s',
                $authArray['data']['tokenType'],
                $authArray['data']['accessToken']
            );
        } else {
            throw new AuthenticationException(__($authArray['message']));
        }
    }

    /**
     * {@inheritDoc}
     */
    public function confirmOrder(string $url, array $data): array
    {
        //This method doesn't need auth, request body has already the API key
        $body = http_build_query($data);
        $rawResponse = $this->post($url, 'confirmOrder', $body);
        return $this->helper->unserializeToArray($rawResponse->getBody());
    }

    /**
     * {@inheritDoc}
     */
    public function getOrderStatus(string $url, array $data): array
    {
        //Authentication first
        if ($this->accessToken === '') {
            $this->authenticate();
        }
        //Then POST call
        $rawResponse = $this->post(
            $url,
            'order/status',
            $this->helper->serializeArray($data),
            'application/json'
        );
        return $this->helper->unserializeToArray($rawResponse->getBody());
    }

    /**
     * {@inheritDoc}
     */
    public function downloadShippingPackaging(string $url, array $data): array
    {
        //Authenticate first
        if ($this->accessToken === '') {
            $this->authenticate();
        }
        //Then GET call
        $rawResponse = $this->get(
            $url,
            'order/downloadLabel',
            $data
        );
        return $this->helper->unserializeToArray($rawResponse->getBody());
    }

    /**
     * {@inheritDoc}
     */
    public function post(
        string $url,
        string $action,
        string $body,
        string $contentType = Client::ENC_URLENCODED
    ): Response {
        //Prepare client
        $this->prepareClient(Request::METHOD_POST, $url, $action);
        //Prepare headers
        $this->prepareHeaders($contentType);
        //Set body inside request
        $this->client->setRawBody($body);
        //Send response
        $response = $this->client->send();
        //Debug
        $this->logCommunication();
        return $response;
    }

    /**
     * {@inheritDoc}
     */
    public function get(string $url, string $action, ?array $params): Response
    {
        //Prepare client
        $this->prepareClient(Request::METHOD_GET, $url, $action);
        //Prepare headers
        $this->prepareHeaders();
        //Encode params
        if ($params !== null) {
            $this->client->setParameterGet($params);
        }
        //Send response
        $response = $this->client->send();
        //Debug
        $this->logCommunication();
        return $response;
    }

    /**
     * Prepare the HTTP client for the request
     *
     * @param string $method
     * @param string $url
     * @param string $action
     */
    protected function prepareClient(string $method, string $url, string $action): void
    {
        $this->client->reset();
        $this->client->setUri($url . $action);
        $this->client->setMethod($method);
        $this->client->setOptions(['timeout' => 30]);
    }

    /**
     * Init headers for the request
     * @param string $contentType
     */
    protected function prepareHeaders(string $contentType = Client::ENC_URLENCODED): void
    {
        //Init headers
        $contentTypeHeader = new ContentType($contentType);
        $accept = new Accept();
        $accept->addMediaType('application/json');
        //Add headers
        $headers = new Headers();
        $headers->addHeader($contentTypeHeader);
        $headers->addHeader($accept);
        //Add authorization header
        if ($this->accessToken !== '') {
            $authorization = new Authorization($this->accessToken);
            $headers->addHeader($authorization);
        }
        $this->client->setHeaders($headers);
    }

    /**
     * Log request and response from HTTP client only in debug mode
     */
    protected function logCommunication(): void
    {
        $this->helper->logDebug('--- [GEL_GATEWAY_REQUEST] ' . $this->client->getLastRawRequest() ?: '');
        $this->helper->logDebug('--- [GEL_GATEWAY_RESPONSE] ' . $this->client->getLastRawResponse() ?: '');
    }
}
