<?php

namespace Drop\GELProximity\Service;

use Drop\GELProximity\Api\ConfigPathInterface;
use Drop\GELProximity\Api\Data\GelShipmentInterface;
use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Api\GELStatusInterface;
use Drop\GELProximity\Api\OrderStatusInterface;
use Drop\GELProximity\Api\Service\GelGatewayInterface;
use Drop\GELProximity\Api\Service\Processors\OrderProcessorInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Helper\Email;
use Drop\GELProximity\Logger\Logger;
use Drop\GELProximity\Model\Carrier\GELProximity;
use Exception;
use InvalidArgumentException;
use Magento\Framework\Exception\CouldNotSaveException;
use Zend\Http\Client\Exception\RuntimeException;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory as OrderCollectionFactory;

/**
 * Class TrackService
 * @package Drop\GELProximity\Service
 */
class TrackService extends AbstractService
{
    /**
     * @var Email
     */
    protected $emailHelper;

    /**
     * @var OrderCollectionFactory
     */
    protected $orderCollectionFactory;

    /**
     * TrackService constructor.
     * @param Data $helper
     * @param GelShipmentRepositoryInterface $gelShipmentRepository
     * @param GelGatewayInterface $gateway
     * @param OrderProcessorInterface $orderProcessor
     * @param Email $emailHelper
     * @param OrderCollectionFactory $orderCollectionFactory
     */
    public function __construct(
        Data $helper,
        GelShipmentRepositoryInterface $gelShipmentRepository,
        GelGatewayInterface $gateway,
        OrderProcessorInterface $orderProcessor,
        Email $emailHelper,
        OrderCollectionFactory $orderCollectionFactory
    ) {
        parent::__construct($helper, $gelShipmentRepository, $gateway, $orderProcessor);
        $this->emailHelper = $emailHelper;
        $this->orderCollectionFactory = $orderCollectionFactory;
    }

    /**
     * Service method that load a collection of order with the Sent to GEL status
     * and make a call to track and change their status based on the current tracking status
     *
     * {@inheritDoc}
     */
    public function run(): int
    {
        if ($this->canBeExecuted()) {
            //Init order collection
            $orderCollection = $this->initServiceCollection();
            //Debug
            $this->helper->logDebug('[TRACK_SERVICE] Starting cron: found ' . $orderCollection->getSize() . ' orders to be processed.');
            /** @var Order $order */
            foreach ($orderCollection as $order) {
                try {
                    //Retrieve the order's shipment status
                    $gelResponse = $this->inquiryOrder($order);
                    //Process the order based on the shipment status
                    $this->processShipmentStatus($order, $gelResponse);
                } catch (InvalidArgumentException $e) {
                    //Catch error in case the deserialization fails
                    $this->helper->log(
                        sprintf(
                            '--- [TRACK_SERVICE] Error for order ID %s during JSON deserialization -> Error: %s',
                            $order->getId(),
                            $e->getMessage()
                        ),
                        Logger::CRITICAL
                    );
                    continue;
                } catch (AuthenticationException $e) {
                    $this->helper->log(
                        sprintf(
                            '--- [TRACK_SERVICE] An error was found when authenticating: %s',
                            $e->getMessage()
                        ),
                        Logger::ERROR
                    );
                } catch (InputException | NoSuchEntityException $e) {
                    //It's impossible that this error gets thrown
                    $this->helper->log(
                        sprintf(
                            '--- [TRACK_SERVICE] Found an unexpected error when loading the GEL shipment for the order ID %s: %s',
                            $order->getId(),
                            $e->getMessage()
                        ),
                        Logger::CRITICAL
                    );
                } catch (RuntimeException $e) {
                    //Possible error in case the endpoint is not reachable (probably timeout)
                    $this->helper->log(
                        sprintf(
                            '--- [TRACK_SERVICE] There\'s an error calling the GEL server: %s',
                            $e->getMessage()
                        ),
                        Logger::ERROR
                    );
                    return 0;
                } catch (CouldNotSaveException $e) {
                    $this->helper->log(
                        sprintf(
                            '--- [TRACK_SERVICE] An error was found for order %s when saving the shipment/tracking: %s',
                            $order->getId(),
                            $e->getMessage()
                        ),
                        Logger::ERROR
                    );
                } catch (Exception $e) {
                    //Probably an error of array undefined index
                    $this->helper->log(
                        sprintf(
                            '--- [TRACK_SERVICE] Found an unexpected error: %s',
                            $e->getMessage()
                        ),
                        Logger::CRITICAL
                    );
                }
            }
            return $orderCollection->getSize();
        } else {
            //Debug
            $this->helper->logDebug('[TRACK_SERVICE] Service not enabled: skipping.');
            return 0;
        }
    }

    /**
     * {@inheritDoc}
     */
    protected function initServiceCollection(): AbstractCollection
    {
        $collection = $this->orderCollectionFactory
            ->create()
            ->addFieldToFilter('main_table.shipping_method', GELProximity::CARRIER_CODE . '_' . GELProximity::CARRIER_CODE)
            ->addFieldToFilter(
                'main_table.' . Order::STATE,
                [
                    'IN' => [
                        OrderStatusInterface::ORDER_STATE_PROCESSING_GEL,
                        OrderStatusInterface::ORDER_STATE_SHIPPED_GEL
                    ]
                ]
            )
            ->addFieldToFilter(
                'main_table.' . Order::STATUS,
                [
                    'IN' => [
                        OrderStatusInterface::ORDER_STATUS_SENT_TO_GEL,
                        OrderStatusInterface::ORDER_STATUS_SHIPPED_GEL
                    ]
                ]
            )
            ->addFieldToFilter('si.state', Order\Invoice::STATE_PAID)
            ->addFieldToFilter(
                'gps.' . GelShipmentInterface::IS_RETURN,
                [
                    ['EQ' => 0],
                    ['null' => true]
                ]
            )
            ->setOrder('main_table.' . Order::ENTITY_ID, Select::SQL_ASC);
        $collection
            ->getSelect()
            ->joinLeft(
                ['si' => $collection->getTable('sales_invoice')],
                'main_table.entity_id = si.order_id',
                []
            )
            ->joinLeft(
                ['gps' => $collection->getTable(GelShipmentInterface::TABLE_NAME)],
                'main_table.' . Order::ENTITY_ID . ' = gps.' . GelShipmentInterface::ORDER_ID,
                [
                    GelShipmentInterface::PICKUP_POINT_ID,
                    GelShipmentInterface::EXTERNAL_ORDER_ID
                ]
            );
        return $collection;
    }

    /**
     * Processes the order data and send it to GEL server
     *
     * @param Order $order
     * @return array
     * @throws AuthenticationException
     * @throws InputException
     * @throws NoSuchEntityException
     * @throws InvalidArgumentException
     */
    protected function inquiryOrder(Order $order): array
    {
        $postUrl = $this->helper->getConfigValue(ConfigPathInterface::URL_GEL_API_SERVER_CONFIG_PATH);
        $gelShipment = $this->gelShipmentRepository->getByOrderId($order->getId());
        //Process order status request's object
        $data = $this->orderProcessor->processTracking($order, $gelShipment);
        //Retrieve order's tracking status
        return $this->gateway->getOrderStatus($postUrl, $data);
    }

    /**
     * Checks the response from GEL server and change the order status
     * based on GEL shipment status, also creates the tracking if exists
     *
     * @param Order $order
     * @param array $response
     * @throws CouldNotSaveException
     */
    protected function processShipmentStatus(Order $order, array $response): void
    {
        //Check success response
        if (array_key_exists('success', $response) && $response['success']) {
            $shipmentStatus = (int)$response['items'][0]['shipment']['status'];
            $tracking = $response['items'][0]['tracking'];
            switch ($shipmentStatus) {
                case GELStatusInterface::SHIPMENT_SHIPPED:
                    //Ship the order only if it isn't already shipped
                    if ($order->getStatus() === OrderStatusInterface::ORDER_STATUS_SENT_TO_GEL) {
                        $this->orderProcessor->ship($order, $shipmentStatus);
                    }
                    //Track the order only if it isn't already tracked and the number is defined
                    if (
                        $tracking['number'] !== null && $tracking['number'] !== ''
                        && !$this->orderIsTracked($order)
                    ) {
                        $this->orderProcessor->track($order, $tracking);
                    }
                    break;
                case GELStatusInterface::SHIPMENT_DELIVERED:
                    //Ship the order only if it isn't already shipped
                    if ($order->getStatus() === OrderStatusInterface::ORDER_STATUS_SENT_TO_GEL) {
                        $this->orderProcessor->ship($order, $shipmentStatus);
                    }
                    //Track the order only if it isn't already tracked and the number is defined
                    if (
                        $tracking['number'] !== null && $tracking['number'] !== ''
                        && !$this->orderIsTracked($order)
                    ) {
                        $this->orderProcessor->track($order, $tracking);
                    }
                    //Complete the order
                    if ($order->getStatus() === OrderStatusInterface::ORDER_STATUS_SHIPPED_GEL) {
                        $this->orderProcessor->complete($order);
                    }
                    break;
            }
            //Debug
            $this->helper->logDebug(
                sprintf(
                    '--- [TRACK_SERVICE] The shipment for order %s is in status %s.',
                    $order->getId(),
                    GELStatusInterface::SHIPMENT_STATUS_LABELS[$shipmentStatus]
                )
            );
        } else {
            //Log gel response error
            $this->helper->log('--- [TRACK_SERVICE] GEL responded with an error: ' . $response['message'], Logger::CRITICAL);
        }
    }

    /**
     * Checks if the order already owns a shipment tracking
     *
     * @param Order $order
     * @return bool
     */
    protected function orderIsTracked(Order $order): bool
    {
        $trackingCollection = $order->getTracksCollection();
        return $trackingCollection->count() > 0;
    }
}
