<?php

namespace Drop\GELProximity\Service;

use Drop\GELProximity\Api\ConfigPathInterface;
use Drop\GELProximity\Api\Data\GelShipmentInterface;
use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Api\OrderStatusInterface;
use Drop\GELProximity\Api\Service\GelGatewayInterface;
use Drop\GELProximity\Api\Service\Processors\OrderProcessorInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Helper\Email;
use Drop\GELProximity\Logger\Logger;
use Drop\GELProximity\Model\Carrier\GELProximity;
use Exception;
use InvalidArgumentException;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory as OrderCollectionFactory;

/**
 * Class ShipService
 * @package Drop\GELProximity\Service
 */
class ShipService extends AbstractService
{
    /**
     * @var Email
     */
    protected $emailHelper;

    /**
     * @var OrderCollectionFactory
     */
    protected $orderCollectionFactory;

    /**
     * ShipService constructor.
     * @param Data $helper
     * @param GelShipmentRepositoryInterface $gelShipmentRepository
     * @param GelGatewayInterface $gateway
     * @param OrderProcessorInterface $orderProcessor
     * @param Email $emailHelper
     * @param OrderCollectionFactory $orderCollectionFactory
     */
    public function __construct(
        Data $helper,
        GelShipmentRepositoryInterface $gelShipmentRepository,
        GelGatewayInterface $gateway,
        OrderProcessorInterface $orderProcessor,
        Email $emailHelper,
        OrderCollectionFactory $orderCollectionFactory
    ) {
        parent::__construct($helper, $gelShipmentRepository, $gateway, $orderProcessor);
        $this->emailHelper = $emailHelper;
        $this->orderCollectionFactory = $orderCollectionFactory;
    }

    /**
     * Service method that load a collection of order with the Processing GEL status
     * and make a call that only ship the order
     *
     * {@inheritDoc}
     */
    public function run(): int
    {
        if ($this->canBeExecuted()) {
            //Init order collection
            $orderCollection = $this->initServiceCollection();
            //Debug
            $this->helper->logDebug('[SHIP_SERVICE] Starting cron: found ' . $orderCollection->getSize() . ' orders to be processed.');
            /** @var Order $order */
            foreach ($orderCollection as $order) {
                //Check first if order can be shipped
                if ($order->canShip()) {
                    //Send data order to GEL server
                    try {
                        $gelResponse = $this->processOrder($order);
                    } catch (InvalidArgumentException $e) {
                        //Catch error in case the deserialization fails
                        $this->helper->log(
                            sprintf(
                                '--- [SHIP_SERVICE] Error for order ID %s during JSON deserialization -> Error: %s',
                                $order->getId(),
                                $e->getMessage()
                            ),
                            Logger::CRITICAL
                        );
                        continue;
                    }
                    //Check for success GEL response
                    if (array_key_exists('success', $gelResponse) && $gelResponse['success']) {
                        try {
                            //Ship order
                            $this->orderProcessor->sentToGEL($order);
                            //Save order and external order IDs references inside shipment
                            $this->updateGelShipment($order, $gelResponse);
                        } catch (CouldNotSaveException $e) {
                            $this->helper->log(
                                sprintf(
                                    '--- [SHIP_SERVICE] Error saving gel shipment entity for order %s -> Error: %s',
                                    $order->getId(),
                                    $e->getMessage()
                                ),
                                Logger::ERROR
                            );
                            //Send email error
                            $this->sendErrorReport($order->getId(), $e->getMessage());
                        } catch (Exception $e) {
                            $this->helper->log(
                                sprintf(
                                    '--- [SHIP_SERVICE] Error saving gel shipment entity for order %s -> Error: %s',
                                    $order->getId(),
                                    $e->getMessage()
                                ),
                                Logger::ERROR
                            );
                        }
                    } else {
                        //Log gel response error
                        $this->helper->log('--- [SHIP_SERVICE] GEL responded with an error: ' . $gelResponse['message'], Logger::CRITICAL);
                        //Send email error
                        $this->sendErrorReport($order->getId(), $gelResponse['message']);
                    }
                } else {
                    //Warn
                    $this->helper->log(
                        '--- [SHIP_SERVICE] Processed order ' . $order->getId() . ' cannot be shipped right now.',
                        Logger::WARNING
                    );
                }
            }
            return $orderCollection->getSize();
        } else {
            //Debug
            $this->helper->logDebug('[SHIP_SERVICE] Service not enabled: skipping.');
            return 0;
        }
    }

    /**
     * {@inheritDoc}
     */
    protected function initServiceCollection(): AbstractCollection
    {
        $collection = $this->orderCollectionFactory
            ->create()
            ->addFieldToFilter('main_table.shipping_method', GELProximity::CARRIER_CODE . '_' . GELProximity::CARRIER_CODE)
            ->addFieldToFilter('main_table.' . Order::STATE, OrderStatusInterface::ORDER_STATE_PROCESSING_GEL)
            ->addFieldToFilter('main_table.' . Order::STATUS, OrderStatusInterface::ORDER_STATUS_PROCESSING_GEL)
            ->addFieldToFilter('si.state', Order\Invoice::STATE_PAID)
            ->addFieldToFilter(
                'gps.' . GelShipmentInterface::IS_RETURN,
                [
                    ['EQ' => 0],
                    ['null' => true]
                ]
            )
            ->setOrder('main_table.' . Order::ENTITY_ID, Select::SQL_ASC);
        $collection
            ->getSelect()
            ->joinLeft(
                ['si' => $collection->getTable('sales_invoice')],
                'main_table.entity_id = si.order_id',
                []
            )
            ->joinLeft(
                ['gps' => $collection->getTable(GelShipmentInterface::TABLE_NAME)],
                'main_table.quote_id = gps.' . GelShipmentInterface::QUOTE_ID,
                [
                    GelShipmentInterface::PICKUP_POINT_ID,
                    GelShipmentInterface::COST,
                    GelShipmentInterface::QUOTE_REFERENCE,
                    GelShipmentInterface::QUOTE_ID
                ]
            );
        return $collection;
    }

    /**
     * Processes the order data and send it to GEL server
     *
     * @param Order $order
     * @return array
     */
    protected function processOrder(Order $order): array
    {
        $postUrl = $this->helper->getConfigValue(ConfigPathInterface::URL_END_USER_BACKEND_CONFIG_PATH);
        $data = $this->orderProcessor->processConfirm($order);
        return $this->gateway->confirmOrder($postUrl, $data);
    }

    /**
     * Retrieve the gel shipment from the quote reference and update with order and external order IDs
     *
     * @param Order $order
     * @param array $gelResponse
     * @throws AlreadyExistsException
     * @throws InputException
     * @throws NoSuchEntityException
     * @return void
     */
    protected function updateGelShipment(Order $order, array $gelResponse): void
    {
        $gelShipment = $this->gelShipmentRepository->getByQuoteReference(
            $order->getData(GelShipmentInterface::QUOTE_REFERENCE)
        );
        //Debug
        $this->helper->logDebug('--- [SHIP_SERVICE] Updating GEL shipment ID ' . $gelShipment->getId());
        $gelShipment->setOrderId($order->getId());
        if (array_key_exists('data', $gelResponse) && array_key_exists('id', $gelResponse['data'])) {
            $gelShipment->setExternalOrderId($gelResponse['data']['id']);
        }
        $this->gelShipmentRepository->save($gelShipment);
        //Debug
        $this->helper->logDebug('--- [SHIP_SERVICE] GEL shipment ID ' . $gelShipment->getId() . ' updated successfully.');
    }

    /**
     * Send the error through an email report
     *
     * @param int $orderId
     * @param string $message
     */
    protected function sendErrorReport(int $orderId, string $message): void
    {
        try {
            $this->emailHelper->sendErrorReport($orderId, $message);
        } catch (MailException | LocalizedException $e) {
            //You must be really unlucky to finish here, anyway, log mail error
            $this->helper->log(
                '--- [SHIP_SERVICE] There was an error sending the error report email: ' . $e->getMessage(),
                Logger::ERROR
            );
        }
    }
}
