<?php

namespace Drop\GELProximity\Model;

use Drop\GELProximity\Api\Data\GelShipmentInterface;
use Drop\GELProximity\Api\Data\GelShipmentSearchResultInterface;
use Drop\GELProximity\Api\Data\GelShipmentSearchResultInterfaceFactory;
use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Model\ResourceModel\GelShipment as GelShipmentResource;
use Drop\GELProximity\Model\ResourceModel\GelShipment\Collection as GelShipmentCollection;
use Drop\GELProximity\Model\ResourceModel\GelShipment\CollectionFactory as GelShipmentCollectionFactory;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SortOrder;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Sales\Api\Data\OrderInterface;

/**
 * Class GelShipmentRepository
 * @package Drop\GELProximity\Model
 */
class GelShipmentRepository implements GelShipmentRepositoryInterface
{
    /**
     * @var GelShipmentResource
     */
    protected $gelShipmentResource;

    /**
     * @var GelShipmentFactory
     */
    protected $gelShipmentFactory;

    /**
     * @var GelShipmentCollectionFactory
     */
    protected $gelShipmentCollectionFactory;

    /**
     * @var GelShipmentSearchResultInterfaceFactory
     */
    protected $gelShipmentSearchResultInterfaceFactory;

    /**
     * @var GelShipmentInterface[]
     */
    protected $registry;

    /**
     * GelShipmentRepository constructor.
     * @param GelShipmentResource $gelShipmentResource
     * @param GelShipmentFactory $gelShipmentFactory
     * @param GelShipmentCollectionFactory $gelShipmentCollectionFactory
     * @param GelShipmentSearchResultInterfaceFactory $gelShipmentSearchResultInterfaceFactory
     * @param array $registry
     */
    public function __construct(
        GelShipmentResource $gelShipmentResource,
        GelShipmentFactory $gelShipmentFactory,
        GelShipmentCollectionFactory $gelShipmentCollectionFactory,
        GelShipmentSearchResultInterfaceFactory $gelShipmentSearchResultInterfaceFactory,
        array $registry = []
    ) {
        $this->gelShipmentResource = $gelShipmentResource;
        $this->gelShipmentFactory = $gelShipmentFactory;
        $this->gelShipmentCollectionFactory = $gelShipmentCollectionFactory;
        $this->gelShipmentSearchResultInterfaceFactory = $gelShipmentSearchResultInterfaceFactory;
        $this->registry = $registry;
    }

    /**
     * {@inheritDoc}
     */
    public function get(int $id): GelShipmentInterface
    {
        if (!$id) {
            throw new InputException(__('An Id is needed. Set the ID and try again.'));
        }
        if (!array_key_exists($id, $this->registry)) {
            $entity = $this->gelShipmentFactory->create();
            $this->gelShipmentResource->load($entity, $id);
            if (!$entity->getEntityId()) {
                throw new NoSuchEntityException(
                    __('The entity that was requested doesn\'t exist. Verify the entity and try again.')
                );
            }
            $this->registry[$id] = $entity;
        }
        return $this->registry[$id];
    }

    /**
     * {@inheritDoc}
     */
    public function getByPickupPointId(int $pickupPointId): GelShipmentInterface
    {
        if (!$pickupPointId) {
            throw new InputException(__('An Id is needed. Set the ID and try again.'));
        }
        $entity = $this->gelShipmentFactory->create();
        $this->gelShipmentResource->load($entity, $pickupPointId, GelShipmentInterface::PICKUP_POINT_ID);
        if (!$entity->getEntityId()) {
            throw new NoSuchEntityException(
                __('The entity that was requested doesn\'t exist. Verify the entity and try again.')
            );
        }
        return $entity;
    }

    /**
     * {@inheritDoc}
     */
    public function getByQuoteReference(string $quoteReference): GelShipmentInterface
    {
        if (!$quoteReference) {
            throw new InputException(__('An Id is needed. Set the ID and try again.'));
        }
        $entity = $this->gelShipmentFactory->create();
        $this->gelShipmentResource->load($entity, $quoteReference, GelShipmentInterface::QUOTE_REFERENCE);
        if (!$entity->getEntityId()) {
            throw new NoSuchEntityException(
                __('The entity that was requested doesn\'t exist. Verify the entity and try again.')
            );
        }
        return $entity;
    }

    /**
     * {@inheritDoc}
     */
    public function getByQuoteId(int $quoteId): GelShipmentInterface
    {
        if (!$quoteId) {
            throw new InputException(__('An Id is needed. Set the ID and try again.'));
        }
        $entity = $this->gelShipmentFactory->create();
        $this->gelShipmentResource->load($entity, $quoteId, GelShipmentInterface::QUOTE_ID);
        if (!$entity->getEntityId()) {
            throw new NoSuchEntityException(
                __('The entity that was requested doesn\'t exist. Verify the entity and try again.')
            );
        }
        return $entity;
    }

    /**
     * {@inheritDoc}
     */
    public function getByQuote(CartInterface $quote): GelShipmentInterface
    {
        return $this->getByQuoteId($quote->getId());
    }

    /**
     * {@inheritDoc}
     */
    public function getByOrderId(int $orderId, bool $isReturn = false): GelShipmentInterface
    {
        if (!$orderId) {
            throw new InputException(__('An Id is needed. Set the ID and try again.'));
        }
        $entity = $this->gelShipmentFactory->create();
        if ($isReturn) {
            //Retrieve the entity ID of the first item that correspond to the filter
            $entityId = $this->gelShipmentCollectionFactory
                ->create()
                ->addFieldToFilter(GelShipmentInterface::ORDER_ID, $orderId)
                ->addFieldToFilter(GelShipmentInterface::IS_RETURN, $isReturn)
                ->getColumnValues(GelShipmentInterface::ENTITY_ID)[0];
            $this->gelShipmentResource->load($entity, $entityId);
        } else {
            $this->gelShipmentResource->load($entity, $orderId, GelShipmentInterface::ORDER_ID);
        }
        if (!$entity->getEntityId()) {
            throw new NoSuchEntityException(
                __('The entity that was requested doesn\'t exist. Verify the entity and try again.')
            );
        }
        return $entity;
    }

    /**
     * {@inheritDoc}
     */
    public function getByOrder(OrderInterface $order, bool $isReturn = false): GelShipmentInterface
    {
        return $this->getByOrderId($order->getEntityId(), $isReturn);
    }

    /**
     * {@inheritDoc}
     * @param GelShipment $gelShipment
     */
    public function save(GelShipmentInterface $gelShipment): GelShipmentInterface
    {
        $this->gelShipmentResource->save($gelShipment);
        $this->registry[$gelShipment->getEntityId()] = $gelShipment;
        return $this->registry[$gelShipment->getEntityId()];
    }

    /**
     * {@inheritDoc}
     * @param GelShipment $gelShipment
     */
    public function delete(GelShipmentInterface $gelShipment): bool
    {
        $this->gelShipmentResource->delete($gelShipment);
        unset($this->registry[$gelShipment->getEntityId()]);
        return true;
    }

    /**
     * {@inheritDoc}
     * @throws InputException
     */
    public function deleteById(int $entityId): bool
    {
        $entity = $this->get($entityId);
        return $this->delete($entity);
    }

    /**
     * {@inheritDoc}
     */
    public function getList(SearchCriteriaInterface $searchCriteria): GelShipmentSearchResultInterface
    {
        $collection = $this->gelShipmentCollectionFactory->create();

        $this->addFiltersToCollection($searchCriteria, $collection);
        $this->addSortOrdersToCollection($searchCriteria, $collection);
        $this->addPagingToCollection($searchCriteria, $collection);

        $collection->load();
        return $this->buildSearchResult($searchCriteria, $collection);
    }

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @param GelShipmentCollection $collection
     */
    protected function addFiltersToCollection(SearchCriteriaInterface $searchCriteria, GelShipmentCollection $collection): void
    {
        foreach ($searchCriteria->getFilterGroups() as $filterGroup) {
            $fields = $conditions = [];
            foreach ($filterGroup->getFilters() as $filter) {
                $fields[] = $filter->getField();
                $conditions[] = [$filter->getConditionType() => $filter->getValue()];
            }
            $collection->addFieldToFilter($fields, $conditions);
        }
    }

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @param GelShipmentCollection $collection
     */
    protected function addSortOrdersToCollection(SearchCriteriaInterface $searchCriteria, GelShipmentCollection $collection): void
    {
        foreach ((array)$searchCriteria->getSortOrders() as $sortOrder) {
            $direction = (string)$sortOrder->getDirection === SortOrder::SORT_ASC ?: SortOrder::SORT_DESC;
            $collection->addOrder($sortOrder->getField(), $direction);
        }
    }

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @param GelShipmentCollection $collection
     */
    protected function addPagingToCollection(SearchCriteriaInterface $searchCriteria, GelShipmentCollection $collection): void
    {
        $collection->setPageSize($searchCriteria->getPageSize());
        $collection->setCurPage($searchCriteria->getCurrentPage());
    }

    /**
     * Build resulting items from collection ahd search criteria
     *
     * @param SearchCriteriaInterface $searchCriteria
     * @param GelShipmentCollection $collection
     * @return GelShipmentSearchResultInterface
     */
    protected function buildSearchResult(
        SearchCriteriaInterface $searchCriteria,
        GelShipmentCollection $collection
    ): GelShipmentSearchResultInterface {
        $searchResults = $this->gelShipmentSearchResultInterfaceFactory->create();
        $searchResults
            ->setSearchCriteria($searchCriteria)
            ->setItems($collection->getItems())
            ->setTotalCount($collection->getSize());
        return $searchResults;
    }
}
