<?php

namespace Drop\GELProximity\Setup;

use Drop\GELProximity\Api\OrderStatusInterface;
use Exception;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Setup\InstallDataInterface;
use Magento\Framework\Setup\ModuleContextInterface;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\StatusFactory;
use Magento\Sales\Model\ResourceModel\Order\Status as StatusResource;
use Symfony\Component\Console\Output\ConsoleOutput;

/**
 * Class InstallData
 * @package Drop\GELProximity\Setup
 */
class InstallData implements InstallDataInterface
{
    /**
     * @var StatusFactory
     */
    protected $orderStatusFactory;

    /**
     * @var StatusResource
     */
    protected $orderStatusResource;

    /**
     * @var ConsoleOutput
     */
    protected $consoleOutput;

    /**
     * InstallData constructor.
     * @param StatusFactory $orderStatusFactory
     * @param StatusResource $orderStatusResource
     * @param ConsoleOutput $consoleOutput
     */
    public function __construct(
        StatusFactory $orderStatusFactory,
        StatusResource $orderStatusResource,
        ConsoleOutput $consoleOutput
    ) {
        $this->orderStatusFactory = $orderStatusFactory;
        $this->orderStatusResource = $orderStatusResource;
        $this->consoleOutput = $consoleOutput;
    }

    /**
     * {@inheritDoc}
     */
    public function install(ModuleDataSetupInterface $setup, ModuleContextInterface $context): void
    {
        $this->addPendingGELStatus();
        $this->addProcessingGELStatus();
    }

    /**
     * Creates the new status Pending GEL
     */
    protected function addPendingGELStatus(): void
    {
        //Create and set data to the new status
        $status = $this->orderStatusFactory->create();
        $status->setStatus(OrderStatusInterface::ORDER_STATUS_PENDING_GEL);
        $status->setData('label', OrderStatusInterface::ORDER_STATUS_PENDING_GEL_LABEL);
        try {
            //Save the new status
            $this->orderStatusResource->save($status);
            //Assign existing state
            $status->assignState(Order::STATE_NEW, false, true);
        } catch (AlreadyExistsException $e) {
            $this->consoleOutput->writeln(
                sprintf(
                    'The status %s already exists. Skipping.',
                    OrderStatusInterface::ORDER_STATUS_PENDING_GEL
                )
            );
        } catch (Exception $e) {
            $this->consoleOutput->writeln(
                sprintf(
                    'There was a problem creating the status %s -> %s',
                    OrderStatusInterface::ORDER_STATUS_PENDING_GEL,
                    $e->getMessage()
                )
            );
        }
    }

    /**
     * Creates the new status Processing GEL
     */
    protected function addProcessingGELStatus(): void
    {
        //Create and set data to the new status
        $status = $this->orderStatusFactory->create();
        $status->setStatus(OrderStatusInterface::ORDER_STATUS_PROCESSING_GEL);
        $status->setData('label', OrderStatusInterface::ORDER_STATUS_PROCESSING_GEL_LABEL);
        try {
            //Save the new status
            $this->orderStatusResource->save($status);
            //Assign existing state
            $status->assignState(OrderStatusInterface::ORDER_STATE_PROCESSING_GEL, true, true);
        } catch (AlreadyExistsException $e) {
            $this->consoleOutput->writeln(
                sprintf(
                    'The status %s already exists. Skipping.',
                    OrderStatusInterface::ORDER_STATUS_PROCESSING_GEL
                )
            );
        } catch (Exception $e) {
            $this->consoleOutput->writeln(
                sprintf(
                    'There was a problem creating the status %s -> %s',
                    OrderStatusInterface::ORDER_STATUS_PROCESSING_GEL,
                    $e->getMessage()
                )
            );
        }
    }
}
