<?php

namespace Drop\GELProximity\Block\Adminhtml\Order\Shipping;

use Drop\GELProximity\Api\Data\GelShipmentInterface;
use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Logger\Logger;
use Drop\GELProximity\Model\Carrier\GELProximity;
use InvalidArgumentException;
use Magento\Backend\Block\Template\Context;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Registry;
use Magento\Sales\Block\Adminhtml\Order\AbstractOrder;
use Magento\Sales\Helper\Admin;

/**
 * Class PickupPoint
 * @package Drop\GELProximity\Block\Adminhtml\Order\Shipping
 */
class PickupPoint extends AbstractOrder
{
    /**
     * @var GelShipmentRepositoryInterface
     */
    protected $gelShipmentRepository;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var GelShipmentInterface|null
     */
    protected $gelShipment = null;

    /**
     * PickupPoint constructor.
     * @param Context $context
     * @param Registry $registry
     * @param Admin $adminHelper
     * @param GelShipmentRepositoryInterface $gelShipmentRepository
     * @param Data $helper
     * @param array $data
     */
    public function __construct(
        Context $context,
        Registry $registry,
        Admin $adminHelper,
        GelShipmentRepositoryInterface $gelShipmentRepository,
        Data $helper,
        array $data = []
    ) {
        parent::__construct($context, $registry, $adminHelper, $data);
        $this->gelShipmentRepository = $gelShipmentRepository;
        $this->helper = $helper;
    }

    /**
     * Returns if the order viewed in admin panel has the GEL shipment
     *
     * @return bool
     */
    public function isGelShipment(): bool
    {
        try {
            $order = $this->getOrder();
        } catch (LocalizedException $e) {
            return false;
        }
        //Gets the shipping method as an object
        $shippingMethod = $order->getShippingMethod(true);
        $isGelShipment = $shippingMethod->getData('carrier_code') === GELProximity::CARRIER_CODE;
        if ($isGelShipment) {
            //Debug
            $this->helper->logDebug(
                sprintf(
                    '[ADMIN_ORDER_VIEW] Processing pickup point view for order %1',
                    $order->getId()
                )
            );
        } else {
            //Debug
            $this->helper->logDebug(
                sprintf(
                    '[ADMIN_ORDER_VIEW] The order %1 doesn\'t have a GEL shipment associated',
                    $order->getId()
                )
            );
        }
        return $isGelShipment;
    }

    /**
     * Returns the GEL shipment for the current order
     *
     * @return GelShipmentInterface|null
     * @throws LocalizedException
     */
    public function getGelShipment(): ?GelShipmentInterface
    {
        if ($this->gelShipment === null) {
            $order = $this->getOrder();
            try {
                $this->gelShipment = $this->gelShipmentRepository->getByQuoteId($order->getQuoteId());
            } catch (InputException | NoSuchEntityException $e) {
                $this->helper->log(
                    Logger::CRITICAL,
                    sprintf(
                        '--- [ADMIN_ORDER_VIEW] There was an error loading the GEL shipment associated to the order %1. Message: %2',
                        $order->getId(),
                        $e->getMessage()
                    )
                );
            }
        }
        return $this->gelShipment;
    }

    /**
     * Formats the additional data of the gel shipment as an address
     *
     * @return string
     */
    public function formatGelShipmentAddress(): string
    {
        try {
            $pickupPointData = $this->helper->unserializeToArray(
                $this->gelShipment->getAdditionalPickupPointData() ?: ''
            );
        } catch (InvalidArgumentException $e) {
            $this->helper->log(
                sprintf(
                    '--- [ADMIN_ORDER_VIEW] There was an error serializing the pickup point data associated to the order %1. Message: %2',
                    $this->gelShipment->getOrderId(),
                    $e->getMessage()
                ),
                Logger::CRITICAL
            );
            return '';
        }
        return sprintf(
            '%s, %s (%s), %s, %s',
            $pickupPointData['address'],
            $pickupPointData['city'],
            $pickupPointData['department'],
            $pickupPointData['zipCode'],
            $pickupPointData['country']
        );
    }
}
