<?php

namespace Drop\GELProximity\Observer;

use Drop\GELProximity\Api\GelShipmentRepositoryInterface;
use Drop\GELProximity\Helper\Data;
use Drop\GELProximity\Logger\Logger;
use Exception;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Sales\Model\Order;

/**
 * Class OrderCancel
 * @package Drop\GELProximity\Observer
 * @observer
 */
class OrderCancel implements ObserverInterface
{
    /**
     * @var GelShipmentRepositoryInterface
     */
    protected $gelShipmentRepository;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * OrderCancel constructor.
     * @param GelShipmentRepositoryInterface $gelShipmentRepository
     * @param Data $helper
     */
    public function __construct(
        GelShipmentRepositoryInterface $gelShipmentRepository,
        Data $helper
    ) {
        $this->gelShipmentRepository = $gelShipmentRepository;
        $this->helper = $helper;
    }

    /**
     * Observer method used to deletes the GEL shipment entity after the order cancellation
     *
     * {@inheritDoc}
     */
    public function execute(Observer $observer)
    {
        if ($this->helper->isModuleEnabled()) {
            /** @var Order $order */
            $order = $observer->getData('order');

            //Debug
            $this->helper->logDebug('[OBSERVER_ORDERCANCEL] Requested cancellation for order ' . $order->getId());
            try {
                //Retrieve the GEL shipment by its quote ID
                $gelShipment = $this->gelShipmentRepository->getByQuoteId($order->getQuoteId());

                //Debug
                $this->helper->logDebug('--- [OBSERVER_ORDERCANCEL] Deleting GEL shipment ' . $gelShipment->getId());

                //Delete the GEL shipment entity from the cancelled order
                $this->gelShipmentRepository->delete($gelShipment);

                //Debug
                $this->helper->logDebug('--- [OBSERVER_ORDERCANCEL] GEL shipment ' . $gelShipment->getId() . ' successfully deleted.');
            } catch (NoSuchEntityException $e) {
                //Log warning
                $this->helper->log(
                    sprintf(
                        '------ [OBSERVER_ORDERCANCEL] No GEL shipment exists for order %s.',
                        $order->getId()
                    ),
                    Logger::WARNING
                );
            } catch (Exception $e) {
                //Log error
                $this->helper->log(
                    sprintf(
                        '------ [OBSERVER_ORDERCANCEL] Encountered an error processing the order %s. Error: %s',
                        $order->getId(),
                        $e->getMessage()
                    ),
                    Logger::ERROR
                );
            }
        } else {
            //Debug
            $this->helper->logDebug('[OBSERVER_ORDERCANCEL] Module not enabled: skipping.');
        }
    }
}
