<?php
/**
 * ConfigObserver
 *
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\FatturazioneElettronica\Observer;

use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Event\Observer as EventObserver;
use Psr\Log\LoggerInterface as Logger;

class ConfigObserver implements ObserverInterface
{
    
    const PREFIX_CONFIG_PATH = 'fatturazione_elettronica/general/prefix';
    
    /**
     * @var Logger
     */
    protected $logger;
    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private $scopeConfig;
    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    private $resourceConnection;

    /**
     * @param Logger $logger
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\App\ResourceConnection $resourceConnection
     */
    public function __construct(
        Logger $logger,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\App\ResourceConnection $resourceConnection
    ) {
        $this->logger = $logger;
        $this->scopeConfig = $scopeConfig;
        $this->resourceConnection = $resourceConnection;
    }

    public function execute(EventObserver $observer)
    {
        $prefixConfigValue = $this->scopeConfig->getValue(self::PREFIX_CONFIG_PATH);

        try {
            $currentInvoicePrefix = $this->getInvoicePrefixSalesSequenceProfile();
            if(empty($currentInvoicePrefix)) {
                $this->logger->info("Invoice prefix fields not found.");
                throw new \Exception("Invoice prefix fields not found.");
            }

            $currentCreditmemoPrefix = $this->getCreditmemoPrefixSalesSequenceProfile();
            if(empty($currentCreditmemoPrefix)) {
                $this->logger->info("Creditmemo prefix fields not found.");
                throw new \Exception("Creditmemo prefix fields not found.");
            }

            if($prefixConfigValue != $currentInvoicePrefix) {
                $this->updateInvoicePrefixSalesSequenceProfile($prefixConfigValue);
            }
            if($prefixConfigValue != $currentCreditmemoPrefix) {
                $this->updateCreditmemoPrefixSalesSequenceProfile($prefixConfigValue);
            }
        } catch (\Exception $e) {
            $this->logger->info(__METHOD__ . ' - ' . $e->getMessage());
        }

        return $this;
    }

    public function getInvoicePrefixSalesSequenceProfile() {
        $invoiceMetaId = $this->getFeSalesSequenceInvoiceMetaId();
        if(empty($invoiceMetaId)) {
            $this->logger->info("Meta_id fields for entity_type=fe_invoice not found.");
            throw new \Exception("Meta_id fields for entity_type=fe_invoice not found.");
        }

        $connection = $this->resourceConnection->getConnection();
        $salesSequenceProfileTable = $this->resourceConnection->getTableName('sales_sequence_profile');
        $select = $connection->select()
            ->from($salesSequenceProfileTable, 'prefix')
            ->where('meta_id = ?', (int) $invoiceMetaId);
        return $connection->fetchOne($select);
    }

    public function getCreditmemoPrefixSalesSequenceProfile() {
        $creditmemoMetaId = $this->getFeSalesSequenceCreditmemoMetaId();
        if(empty($creditmemoMetaId)) {
            $this->logger->info("Meta_id fields for entity_type=fe_creditmemo not found.");
            throw new \Exception("Meta_id fields for entity_type=fe_creditmemo not found.");
        }

        $connection = $this->resourceConnection->getConnection();
        $salesSequenceProfileTable = $this->resourceConnection->getTableName('sales_sequence_profile');
        $select = $connection->select()
            ->from($salesSequenceProfileTable, 'prefix')
            ->where('meta_id = ?', (int) $creditmemoMetaId);
        return $connection->fetchOne($select);
    }

    public function updateInvoicePrefixSalesSequenceProfile($prefix) {
        $invoiceMetaId = $this->getFeSalesSequenceInvoiceMetaId();
        if(empty($invoiceMetaId)) {
            $this->logger->info("Meta_id fields for entity_type=fe_invoice not found.");
            throw new \Exception("Meta_id fields for entity_type=fe_invoice not found.");
        }

        $connection = $this->resourceConnection->getConnection();
        $salesSequenceProfileTable = $this->resourceConnection->getTableName('sales_sequence_profile');
        $connection->update(
            $salesSequenceProfileTable,
            ['prefix' => $prefix],
            ['meta_id = ?' => (int) $invoiceMetaId]
        );
        return $this;
    }

    public function updateCreditmemoPrefixSalesSequenceProfile($prefix) {
        $creditmemoMetaId = $this->getFeSalesSequenceCreditmemoMetaId();
        if(empty($creditmemoMetaId)) {
            $this->logger->info("Meta_id fields for entity_type=fe_creditmemo not found.");
            throw new \Exception("Meta_id fields for entity_type=fe_creditmemo not found.");
        }

        $connection = $this->resourceConnection->getConnection();
        $salesSequenceProfileTable = $this->resourceConnection->getTableName('sales_sequence_profile');
        $connection->update(
            $salesSequenceProfileTable,
            ['prefix' => $prefix],
            ['meta_id = ?' => (int) $creditmemoMetaId]
        );
        return $this;
    }

    public function getFeSalesSequenceInvoiceMetaId() {
        $connection = $this->resourceConnection->getConnection();
        $salesSequenceMetaTable = $this->resourceConnection->getTableName('sales_sequence_meta');
        $select = $connection->select()
            ->from($salesSequenceMetaTable, 'meta_id')
            ->where('entity_type = ?', 'fe_invoice');
        return $connection->fetchOne($select);
    }

    public function getFeSalesSequenceCreditmemoMetaId() {
        $connection = $this->resourceConnection->getConnection();
        $salesSequenceMetaTable = $this->resourceConnection->getTableName('sales_sequence_meta');
        $select = $connection->select()
            ->from($salesSequenceMetaTable, 'meta_id')
            ->where('entity_type = ?', 'fe_creditmemo');
        return $connection->fetchOne($select);
    }
}
