<?php
/**
 * CheckGridsIntegrity
 *
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\FatturazioneElettronica\Cron;

use \Magento\Framework\App\ResourceConnection;
use \Magento\Framework\DB\Adapter\AdapterInterface;
use \Drop\FatturazioneElettronica\Logger\Logger;

class CheckGridsIntegrity
{

    protected $invoiceMainTableName = 'sales_invoice';
    protected $invoiceGridTableName = 'sales_invoice_grid';
    protected $creditmemoMainTableName = 'sales_creditmemo';
    protected $creditmemoGridTableName = 'sales_creditmemo_grid';
    protected $connection;

    /**
     * @var ResourceConnection
     */
    protected $resourceConnection;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * CheckGridsIntegrity constructor.
     * @param ResourceConnection $resourceConnection
     * @param Logger $logger
     */
    public function __construct(
        ResourceConnection $resourceConnection,
        Logger $logger
    ) {
        $this->resourceConnection = $resourceConnection;
        $this->logger = $logger;
    }

    /**
     * Refresh fe_increment_id field of grid tables if is missed
     * @return $this
     * @throws \Zend_Db_Statement_Exception
     * @todo document_type sync
     */
    public function execute() {
        try {
            $result = $this->refreshFeIncrementId($this->invoiceMainTableName, $this->invoiceGridTableName);
            $this->logger->info("Refresh {$this->invoiceGridTableName}.fe_increment_id affected rows: " . $result->rowCount());

            $result = $this->refreshFeIncrementId($this->creditmemoMainTableName, $this->creditmemoGridTableName);
            $this->logger->info("Refresh {$this->creditmemoGridTableName}.fe_increment_id affected rows: " . $result->rowCount());
        } catch (Exception $e) {
            $this->logger->error("Refresh fe_increment_id Exception throwed: " . $e->getMessage());
        }
        return $this;
    }

    /**
     * Select grid null field, not null main field and insert on duplicate update
     * @param $mainTableName
     * @param $gridTableName
     * @return \Zend_Db_Statement_Interface
     */
    public function refreshFeIncrementId($mainTableName, $gridTableName) {
        $select = $this->getConnection()->select()
            ->from([$gridTableName => $this->getTable($gridTableName)], ['entity_id', 'order_id', 'customer_name'])
            ->joinLeft(
                [$mainTableName => $this->getTable($mainTableName)],
                sprintf(
                    '%s.%s = %s.%s',
                    $gridTableName,
                    'increment_id',
                    $mainTableName,
                    'increment_id'
                ),
                ['fe_increment_id']
            )
            ->where($gridTableName . '.fe_increment_id' . ' IS NULL')
            ->where($mainTableName . '.fe_increment_id' . ' IS NOT NULL');

        $insert = $this->getConnection()->insertFromSelect(
            $select,
            $this->getTable($gridTableName),
            ['entity_id', 'order_id', 'customer_name', 'fe_increment_id'],
            AdapterInterface::INSERT_ON_DUPLICATE
        );
        return $this->getConnection()->query($insert);
    }

    /**
     * Returns connection
     *
     * @return AdapterInterface
     */
    public function getConnection()
    {
        if (!$this->connection) {
            $this->connection = $this->resourceConnection->getConnection('sales');
        }
        return $this->connection;
    }

    /**
     * Returns table names
     * @param $tableName
     * @return string
     */
    public function getTable($tableName) {
        return $this->resourceConnection->getTableName($tableName);
    }

}
