<?php

namespace Drop\FatturazioneElettronica\Observer\Order;

class InvoicePay implements \Magento\Framework\Event\ObserverInterface
{

    const FE_INVOICE_ENTITY_TYPE = 'fe_invoice';
    const FE_INVOICE_STORE_ID = 0;

    const FE_INVOICE_INVOICE_TYPE = 'Accompagnatoria';
    const FE_INVOICE_PROFORMA_TYPE = 'Proforma';

    const XML_EU_COUNTRIES_PATH = 'general/country/eu_countries';
    const DIGITS_CONFIG_PATH = 'fatturazione_elettronica/general/digits';
    const PREFIX_CONFIG_PATH = 'fatturazione_elettronica/general/prefix';

    /**
     * @var \Magento\SalesSequence\Model\Manager
     */
    protected $sequenceManager;
    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private $scopeConfig;
    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;
    /**
     * @var \Drop\FatturazioneElettronica\Model\TaxvatNumerationFactory
     */
    private $taxvatFactory;
    /**
     * @var \Magento\SalesSequence\Model\ResourceModel\Meta
     */
    private $meta;
    /**
     * @var \Magento\SalesSequence\Model\ResourceModel\Profile
     */
    private $profile;

    /**
     * InvoicePay constructor.
     * @param \Magento\SalesSequence\Model\Manager $sequenceManager
     * @param \Psr\Log\LoggerInterface $logger
     */
    public function __construct(
        \Magento\SalesSequence\Model\Manager $sequenceManager,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Drop\FatturazioneElettronica\Model\TaxvatNumerationFactory $taxvatFactory,
        \Magento\SalesSequence\Model\ResourceModel\Meta $meta,
        \Magento\SalesSequence\Model\ResourceModel\Profile $profile,
        \Psr\Log\LoggerInterface $logger
    )
    {
        $this->sequenceManager = $sequenceManager;
        $this->scopeConfig = $scopeConfig;
        $this->logger = $logger;
        $this->taxvatFactory = $taxvatFactory;
        $this->meta = $meta;
        $this->profile = $profile;
    }

    public function execute(\Magento\Framework\Event\Observer $observer)
    {
        $invoice = $observer->getEvent()->getInvoice();
        if (!$invoice) {
            $this->logger->debug(__METHOD__ . ' - Invoice not found.');
            return false;
        }

        $order = $invoice->getOrder();
        if (!$order) {
            $this->logger->debug(__METHOD__ . ' - Order not found.');
            return false;
        }

        $billingAddress = $order->getShippingAddress();
        if (!$billingAddress) {
            $this->logger->debug(__METHOD__ . ' - Order Billing Address not found.');
            return false;
        }

        $euCountries = explode(',', $this->scopeConfig->getValue(self::XML_EU_COUNTRIES_PATH));

        $invoice->setDocumentType(self::FE_INVOICE_PROFORMA_TYPE);
        if ($billingAddress->getRequestInvoice()) {
            try {
                $docType = in_array($billingAddress->getCountryId(), $euCountries) ? self::FE_INVOICE_PROFORMA_TYPE : self::FE_INVOICE_INVOICE_TYPE;

                $customTaxvatNumeration = $this->taxvatFactory->create()
                    ->getCollection()
                    ->addFieldToFilter('countries', ['like' => '%' . $billingAddress->getCountryId() . '%'])
                    ->getFirstItem();
                $invoiceEntityType = self::FE_INVOICE_ENTITY_TYPE.'_'.$customTaxvatNumeration->getId();

                $nextValue = $this->sequenceManager->getSequence($invoiceEntityType, self::FE_INVOICE_STORE_ID)->getNextValue();
                $nextValue = $this->setNextValueLenght($invoiceEntityType, $nextValue);
                $invoice->setFeIncrementId($nextValue)
                    ->setDocumentType($docType)
                    ->addComment('Custom fe_increment_id: #' . $nextValue);
            } catch (\Exception $e) {
                $this->revertLastGeneratedSequenceNumber($nextValue);
                $this->logger->error("Failed to set Custom fe_increment_id to invoice {$invoice->getIncrementId()}: " . $e->getMessage());
                return false;
            }

            try {
                $order->addStatusHistoryComment('Custom invoice fe_increment_id: #' . $nextValue)->save();
            } catch (\Exception $e) {
                $this->logger->error("Failed to set History comment to order {$order->getIncrementId()}: " . $e->getMessage());
            }
        }

        return $this;
    }

    /**
     * @param $lastGeneratedNumber
     */
    private function revertLastGeneratedSequenceNumber($lastGeneratedNumber)
    {
        //Aggiungere metodo resetLastValue a \Magento\SalesSequence\Model\Sequence che fa queste query:
        //DELETE FROM sequence_fe_invoice_0 WHERE sequence_value = $lastGeneratedNumber;
        //ALTER TABLE sequence_fe_invoice_0 AUTO_INCREMENT = $lastGeneratedNumber;
        $this->logger->error(__METHOD__ . ' TODO');
        return $this;
    }

    public function setNextValueLenght($invoiceEntityType, $nextValue)
    {
        $digitsConfigValue = $this->scopeConfig->getValue(self::DIGITS_CONFIG_PATH);

        $meta = $this->meta->loadByEntityTypeAndStore($invoiceEntityType, self::FE_INVOICE_STORE_ID);
        $profile = $this->profile->loadActiveProfile($meta->getMetaId());
        $prefixConfigValue = $profile->getPrefix();

        $nextValue = str_replace($prefixConfigValue, '', $nextValue);
        if ($digitsConfigValue > strlen($nextValue)) {
            $zeroAddString = '';
            for ($i = 0; $i < ($digitsConfigValue - strlen($nextValue)); $i++) {
                $zeroAddString .= '0';
            }
            $nextValue = $zeroAddString . $nextValue;
        } elseif ($digitsConfigValue < strlen($nextValue)) {
            $nextValue = substr($nextValue, (strlen($nextValue) - $digitsConfigValue));
        }
        return ($prefixConfigValue . $nextValue);
    }
}
