<?php
/**
 * ResetSequence
 *
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\FatturazioneElettronica\Cron;

class ResetSequence
{
    const SEQUENCE_STORE_ID = 0;
    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    private $resourceConnection;
    /**
     * @var \Drop\FatturazioneElettronica\Logger\Logger
     */
    private $logger;
    /**
     * @var \Magento\SalesSequence\Model\ResourceModel\Meta
     */
    private $meta;
    /**
     * @var \Magento\SalesSequence\Model\ResourceModel\Profile
     */
    private $profile;
    /**
     * @var \Magento\Framework\Stdlib\DateTime\TimezoneInterface 
     */
    private $timezone;
    /**
     * @var \Webkul\Grid\Model\GridFactory
     */
    var $taxvatFactory;

    /**
     * ResetSequence constructor.
     * @param \Magento\Framework\App\ResourceConnection $resourceConnection
     * @param \Drop\FatturazioneElettronica\Logger\Logger $logger
     * @param \Magento\SalesSequence\Model\ResourceModel\Meta $meta
     * @param \Magento\SalesSequence\Model\ResourceModel\Profile $profile
     * @param \Drop\FatturazioneElettronica\Model\TaxvatNumerationFactory $taxvatFactory
     */
    public function __construct(
        \Magento\Framework\App\ResourceConnection $resourceConnection,
        \Drop\FatturazioneElettronica\Logger\Logger $logger,
        \Magento\SalesSequence\Model\ResourceModel\Meta $meta,
        \Magento\SalesSequence\Model\ResourceModel\Profile $profile,
        \Drop\FatturazioneElettronica\Model\TaxvatNumerationFactory $taxvatFactory,
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $timezone
    )
    {
        $this->resourceConnection = $resourceConnection;
        $this->logger = $logger;
        $this->meta = $meta;
        $this->profile = $profile;
        $this->taxvatFactory = $taxvatFactory;
        $this->timezone = $timezone;
    }

    /**
     * @return $this
     */
    public function execute()
    {
        $model = $this->taxvatFactory->create();
        $collection = $model->getCollection();
        foreach ($collection as $item) {
            $model->load($item->getId());
            $prefixConfigValue = $this->updatePrefixToLatestYear($item->getCustomNumeration());
            try {
                // update table drop_efatt_taxvatnumeration
                $model->setCustomNumeration($prefixConfigValue);
                $model->save();
            } catch (\Exception $e) {
                $this->logger->error('ERROR updating sales sequence prefix: ' . $e->getMessage());
            }

            // update sequence table
            $entityTypes = ['fe_invoice_' . $item->getId(), 'fe_creditmemo_' . $item->getId()];
            foreach ($entityTypes as $entityType) {
                $meta = $this->meta->loadByEntityTypeAndStore($entityType, self::SEQUENCE_STORE_ID);
                $profile = $this->profile->loadActiveProfile($meta->getMetaId());
                $profile->setPrefix($prefixConfigValue);
                try {
                    $profile->save();
                } catch (\Exception $e) {
                    $this->logger->error("ERROR update sequence profile: " . $e->getMessage());
                }

                // truncate sequence table
                $this->resetTable($meta->getSequenceTable());
            }

        }

        return $this;
    }

    /**
     * @param $oldPrefixValue
     * @return mixed
     */
    public function updatePrefixToLatestYear($oldPrefixValue)
    {
        $lastYear = $this->timezone->date('-1 year')->format('Y');
        $currentYear = $this->timezone->date()->format('Y');
        $newYearPrefixValue = str_replace($lastYear, $currentYear, $oldPrefixValue);

        return $newYearPrefixValue;
    }

    /**
     * @param $table
     */
    public function resetTable($table)
    {
        try {
            $connection = $this->resourceConnection->getConnection();
            $tableName = $this->resourceConnection->getTableName($table);
            $connection->truncateTable($tableName);
        } catch (\Exception $e) {
            $this->logger->error("ERROR truncate table: " . $e->getMessage());
        }
    }

}
