<?php
/**
 * ResetSequence
 *
 * @copyright Copyright © 2018 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\FatturazioneElettronica\Cron;

use Magento\Framework\Exception\EmailNotConfirmedException;
use \Magento\Sales\Model\ResourceModel\Order\Invoice\CollectionFactory as InvoiceCollectionFactory;
use \Magento\Sales\Model\ResourceModel\Order\Creditmemo\CollectionFactory as CreditmemoCollectionFactory;
use \Magento\Framework\App\Config\ScopeConfigInterface;
use \Magento\Store\Model\StoreManagerInterface;
use \Magento\Framework\Translate\Inline\StateInterface;
use \Magento\Framework\Mail\Template\TransportBuilder;
use \Drop\FatturazioneElettronica\Logger\Logger;

class CheckSequence
{

    const XML_DOCUMENT_PREFIX_PATH = 'fatturazione_elettronica/general/prefix';
    const XML_DEBUG_EMAIL = 'fatturazione_elettronica/general/debug_email';
    const XML_GENERAL_SENDER_EMAIL = 'trans_email/ident_general/email';
    const XML_GENERAL_SENDER_NAME = 'trans_email/ident_general/name';

    /**
     * @var InvoiceCollectionFactory
     */
    protected $invoiceCollectionFactory;

    /**
     * @var CreditmemoCollectionFactory
     */
    protected $creditmemoCollectionFactory;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var TransportBuilder
     */
    protected $transportBuilder;

    /**
     * @var StateInterface
     */
    protected $inlineTranslation;

    /**
     * CheckSequence constructor.
     */
    public function __construct(
        InvoiceCollectionFactory $invoiceCollectionFactory,
        CreditmemoCollectionFactory $creditmemoCollectionFactory,
        ScopeConfigInterface $scopeConfig,
        StoreManagerInterface $storeManager,
        TransportBuilder $transportBuilder,
        StateInterface $inlineTranslation,
        Logger $logger
    ) {
        $this->invoiceCollectionFactory = $invoiceCollectionFactory;
        $this->creditmemoCollectionFactory = $creditmemoCollectionFactory;
        $this->logger = $logger;
        $this->scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
        $this->transportBuilder = $transportBuilder;
        $this->inlineTranslation = $inlineTranslation;
    }

    public function execute() {
        $errorMessages = '';

        try {
            $errorMessages .= $this->getInvoicesAreConsecutive();
        } catch (Exception $e) {
            $this->logger->error($e->getMessage());
        }

        try {
            $errorMessages .= $this->getCreditmemosAreConsecutive();
        } catch (Exception $e) {
            $this->logger->error($e->getMessage());
        }

        if(empty($errorMessages)) {
            return $this;
        }

        try {
            $this->sendLogMail([
                'title' => 'Check sequence error:',
                'message' => $errorMessages
            ]);
            $this->logger->error($errorMessages);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }
        return $this;
    }

    private function getCreditmemosAreConsecutive() {
        $creditmemosCollection = $this->creditmemoCollectionFactory->create()
            ->addAttributeToFilter('fe_increment_id', ['notnull' => true])
            ->setOrder('created_at','ASC');
        if($creditmemosCollection->count() == 0) {
            return;
        }

        $numericIncrementIds = [];
        foreach($creditmemosCollection as $creditmemo) {
            $numericIncrementIds['#' . $creditmemo->getIncrementId()] = ((int) str_replace($this->getPrefix(), '', $creditmemo->getFeIncrementId()));
        }

        asort($numericIncrementIds);

        $i = 1;
        $errors = '';
        foreach($numericIncrementIds as $incrementId => $feIncrementId) {
            if($feIncrementId != $i) {
                $errors .= "<br/>fe_increment_id of INVOICE {$incrementId} is not consecutive with the previous one.<br/>";
            }
            $i++;
        }

        if(empty($errors)) {
            $this->logger->info('Check CREDITMEMO sequence script completed correctly');
            return;
        }
        return $errors;
    }

    private function getInvoicesAreConsecutive() {
        $invoicesCollection = $this->invoiceCollectionFactory->create()
            ->addAttributeToFilter('fe_increment_id', ['notnull' => true])
            ->setOrder('created_at','ASC');
        if($invoicesCollection->count() == 0) {
            return;
        }

        $numericIncrementIds = [];
        foreach($invoicesCollection as $invoice) {
            $numericIncrementIds['#' . $invoice->getIncrementId()] = ((int) str_replace($this->getPrefix(), '', $invoice->getFeIncrementId()));
        }

        asort($numericIncrementIds);

        $i = 1;
        $errors = '';
        foreach($numericIncrementIds as $incrementId => $feIncrementId) {
            if($feIncrementId != $i) {
                $errors .= "<br/>fe_increment_id of INVOICE {$incrementId} is not consecutive with the previous one.<br/>";
            }
            $i++;
        }

        if(empty($errors)) {
            $this->logger->info('Check INVOICE sequence script completed correctly');
            return;
        }

        return $errors;
    }

    private function getPrefix() {
        return $this->scopeConfig->getValue(self::XML_DOCUMENT_PREFIX_PATH);
    }

    public function sendLogMail($templateVars){
        $to = $this->scopeConfig->getValue(self::XML_DEBUG_EMAIL);
        if(empty($to)) {
            $this->logger->error('Cannot send mail log because Debug Email configuration is empty');
        }

        try {
            $this->inlineTranslation->suspend();
            $transport = $this->transportBuilder
                ->setTemplateIdentifier('fe_log')
                ->setTemplateOptions(['area' => \Magento\Framework\App\Area::AREA_FRONTEND, 'store' => $this->storeManager->getStore()->getId()])
                ->setTemplateVars($templateVars)
                ->setFrom([
                    'email' => $this->scopeConfig->getValue(self::XML_GENERAL_SENDER_EMAIL),
                    'name' => $this->scopeConfig->getValue(self::XML_GENERAL_SENDER_NAME)
                ])
                ->addTo($to)
                ->getTransport();
            $transport->sendMessage();
            $this->inlineTranslation->resume();
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
            return false;
        }
        return true;
    }

}
