<?php
/**
 * AwsS3
 *
 * @copyright Copyright © 2019 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\FatturazioneElettronica\Model\Destination;

use \Drop\FatturazioneElettronica\Model\Config\Source\Destination;
use \Aws\S3\Exception\S3Exception;

class AwsS3
{

    const XML_UPLOAD_FE_DESTINATION = 'fatturazione_elettronica/upload_fe/destination';
    const XML_AWS_S3_BUCKET_NAME = 'fatturazione_elettronica/upload_fe/aws_s3_bucket_name';
    const XML_AWS_S3_BUCKET_REGION = 'fatturazione_elettronica/upload_fe/aws_s3_bucket_region';
    const XML_AWS_S3_IAM_KEY = 'fatturazione_elettronica/upload_fe/aws_s3_iam_key';
    const XML_AWS_S3_IAM_SECRET = 'fatturazione_elettronica/upload_fe/aws_s3_iam_secret';

    /**
     * @var \Drop\FatturazioneElettronica\Logger\Logger
     */
    protected $logger;

    /**
     * @var Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTime
     */
    protected $dateTime;

    /**
     * @var \Aws\S3\S3Client $s3Client
     */
    public function __construct(
        \Drop\FatturazioneElettronica\Logger\Logger $logger,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\Stdlib\DateTime\DateTime $dateTime
    ) {
        $this->logger = $logger;
        $this->scopeConfig = $scopeConfig;
        $this->dateTime = $dateTime;
    }

    /**
     * Connect to AWS function. @todo: Get it work with Dependency Injection
     * @return \Aws\S3\S3Client
     */
    private function connect() {
        return \Aws\S3\S3Client::factory([
            'credentials' => [
                'key' => $this->getIamKey(),
                'secret' => $this->getIamSecret()
            ],
            'version' => 'latest',
            'region'  => $this->getBucketRegion()
        ]);
    }

    /**
     * Save file to AWS
     * @param $fileInfo
     * @return $this
     */
    public function saveFiles($fileInfo) {
        if(!$this->getUploadDestinationIsAws()) {
            return $this;
        }
        if(!$this->getUploadDestinationHasConnectionData()) {
            $this->logger->error('Upload to AWS is enabled but missing required connection data. Exit.');
            return $this;
        }
        if(empty($fileInfo)) {
            $this->logger->error('Empty file to send to S3');
            return $this;
        }

        try {
            $connection = $this->connect();
        } catch (Exception $e) {
            $this->logger->error("Can't connect to AWS S3: " . $e->getMessage());
            return $this;
        }

        foreach($fileInfo as $fileName => $fileContent) {
            if(empty($fileName) || empty($fileContent)) {
                $this->logger->error('Empty file to send to S3');
                continue;
            }

            try {
                $connection->putObject([
                    'Bucket'=> $this->getBucketName(),
                    'Key' =>  date('Y-m-d', ($this->dateTime->timestamp() + $this->dateTime->getGmtOffset())) . '/' . $fileName,
                    'Body' => $fileContent,
                    'StorageClass' => 'REDUCED_REDUNDANCY'
                ]);
            } catch (S3Exception $e) {
                $this->logger->error($e->getMessage());
            } catch (\Exception $e) {
                $this->logger->error($e->getMessage());
            }
        }

        return $this;
    }

    private function getUploadDestinationIsAws() {
        if($this->getUploadDestination() == Destination::DESTINATION_AWS_S3) {
            return true;
        }
        return false;
    }
    private function getUploadDestinationHasConnectionData() {
        if(empty($this->getBucketName()) || empty($this->getBucketRegion()) || empty($this->getIamKey()) || empty($this->getIamSecret())) {
            return false;
        }
        return true;
    }
    private function getUploadDestination() {
        return $this->scopeConfig->getValue(self::XML_UPLOAD_FE_DESTINATION);
    }
    private function getBucketName() {
        return $this->scopeConfig->getValue(self::XML_AWS_S3_BUCKET_NAME);
    }
    private function getBucketRegion() {
        return $this->scopeConfig->getValue(self::XML_AWS_S3_BUCKET_REGION);
    }
    private function getIamKey() {
        return $this->scopeConfig->getValue(self::XML_AWS_S3_IAM_KEY);
    }
    private function getIamSecret() {
        return $this->scopeConfig->getValue(self::XML_AWS_S3_IAM_SECRET);
    }

}
