<?php

namespace Drop\FatturazioneElettronica\Helper;

use Magento\Framework\App\Filesystem\DirectoryList;
use \Drop\FatturazioneElettronica\Model\FixSequence as Data;

class FixSequence extends \Magento\Framework\App\Helper\AbstractHelper
{
    const XML_SEQUENCE_FIX_ENABLED_PATH = 'fatturazione_elettronica/general/enable_sequence_fix';
    const XML_FE_INCREMENT_ID_DIGITS_NUMBER_PATH = 'fatturazione_elettronica/general/digits';
    const XML_EU_COUNTRIES_PATH = 'general/country/eu_countries';
    const SEQUENCE_LOG_FILE = 'fe_sequence.log';
    const OLD_TAX_REPRESENTATION = ['FR', 'DE', 'AT', 'BE'];
    const TAX_REFORM_BREAKDATE = "2021-07-01 00:00:00";
    const TAX_REFORM_YEAR = "21";

    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    private $_resource;

    /**
     * @var \Drop\FatturazioneElettronica\Model\ResourceModel\TaxvatNumeration\CollectionFactory
     */
    private $_taxVatCollection;

    /**
     * @var \Magento\Framework\Filesystem
     */
    private $_filesystem;

    /**
     * @var ScopeConfigInterface
     */
    private $_scopeConfig;

    /**
     * @var \Magento\Framework\Filesystem\Driver\File
     */
    private $_fileDriver;

    /**
     * @var \Magento\Framework\DB\Adapter\AdapterInterface
     */
    private $_connection;

    private $logger;

    private $queryToRun = [];

    /**
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Framework\App\ResourceConnection $resource
     * @param \Drop\FatturazioneElettronica\Model\ResourceModel\TaxvatNumeration\CollectionFactory $taxVatCollection
     * @param \Magento\Framework\Filesystem $filesystem
     * @param \Magento\Framework\Filesystem\Driver\File $fileDriver
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Framework\App\ResourceConnection $resource,
        \Drop\FatturazioneElettronica\Model\ResourceModel\TaxvatNumeration\CollectionFactory $taxVatCollection,
        \Magento\Framework\Filesystem $filesystem,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\Filesystem\Driver\File $fileDriver
    )
    {
        $this->_resource = $resource;
        $this->_connection = $resource->getConnection();
        $this->_taxVatCollection = $taxVatCollection;
        $this->_filesystem = $filesystem;
        $this->_scopeConfig = $scopeConfig;
        $this->_fileDriver = $fileDriver;

        $writer = new \Zend\Log\Writer\Stream(BP . '/var/log/' . self::SEQUENCE_LOG_FILE);
        $this->logger = new \Zend\Log\Logger();
        $this->logger->addWriter($writer);

        parent::__construct($context);
    }

    /**
     * @param null $dateIni
     * @param null $dateEnd
     * @param null $customNumPrefix
     * @param null $iniNumber
     * @param null $docType
     * @throws \Exception
     */
    public function run($dateIni = null, $dateEnd = null, $customNumPrefix = null, $iniNumber = null, $docType = null)
    {
        if (!$this->isSequenceFixEnabled()) {
            return;
        }

        $this->logger->debug('inzio fix sequence ' . date('Y-m-d H:i'));

        $conn = $this->_connection->getConnection();
        if (!$dateIni || !$dateEnd) {
            $dateIni = date('Y-m-d', strtotime("-1 days"));
            $dateEnd = date('Y-m-d');
        }
        $this->logger->debug('periodo: ' . $dateIni . ' - ' . $dateEnd);

        foreach ($this->_taxVatCollection->create() as $customTaxVat) {
            $customNumeration = $customTaxVat->getCustomNumeration();
            $explCustomNumeration = explode('-', $customNumeration);
            if (!isset($explCustomNumeration[1])) {
                continue;
            }
            if ($customNumPrefix && strpos($customNumeration, $customNumPrefix) === false) {
                continue;
            }
            $countryCode = str_replace(self::TAX_REFORM_YEAR, "", $explCustomNumeration[1]);

            $this->logger->debug('sequence ' . $customNumeration);

            foreach (Data::TABLE as $table => $tables) {
                $sequenceTable = Data::SEQUENCE[$table][0] . $customTaxVat->getId() . '_0';
                $sequenceCreatedAt = $this->getSequenceCreatedAt($sequenceTable);
                $table = $this->_connection->getTableName($table);
                if ($docType && strpos($table, $docType) === false) {
                    continue;
                }

                try {
                    if (in_array($countryCode, self::OLD_TAX_REPRESENTATION) && $table == 'sales_creditmemo') {
                        // gestione documenti per ordini PRE riforma fiscale 1 luglio
                        $docs = $this->getDocuments($conn, $table, $customTaxVat, $countryCode, $sequenceCreatedAt, $dateIni, $dateEnd, true);
                        $this->logger->debug('table ' . $table . ' total: ' . count($docs));
                        $this->manageDocuments($table, $docs, $customNumeration, $iniNumber);

                        // gestione documenti per ordini POST riforma fiscale 1 luglio
                        $docs = $this->getDocuments($conn, $table, $customTaxVat, $countryCode, $sequenceCreatedAt, $dateIni, $dateEnd);
                        $this->logger->debug('table ' . $table . ' total: ' . count($docs));
                        $lastIncrementChecked = $this->manageDocuments($table, $docs, $customNumeration, $iniNumber);

                    } else {
                        $docs = $this->getDocuments($conn, $table, $customTaxVat, $countryCode, $sequenceCreatedAt, $dateIni, $dateEnd);

                        $this->logger->debug('table ' . $table . ' total: ' . count($docs));

                        $lastIncrementChecked = $this->manageDocuments($table, $docs, $customNumeration, $iniNumber);

                        $this->queryToRun[] = $this->checkSequence($sequenceTable, $lastIncrementChecked);
                    }

                } catch (\Exception $e) {
                    throw $e;
                }
            }
        }

//        echo "<pre>";
//        print_r($this->queryToRun);
//        echo "</pre>";
//        die();

        if (!count($this->queryToRun)) {
            return;
        }

        $conn->beginTransaction();

        try {
            foreach ($this->queryToRun as $q) {
                $conn->query(
                    $q
                );
            }
        } catch (\Exception $e) {
            $conn->rollBack();
            throw $e;
        }

        $conn->commit();

        $this->logger->debug('fine fix sequence ' . date('Y-m-d H:i'));
    }

    /**
     * @param $table
     * @param $docs
     * @param $customNumeration
     * @param , $iniNumber
     * @return int|mixed
     */
    private function manageDocuments($table, $docs, $customNumeration, $iniNumber)
    {
        $lastIncrementChecked = 0;
        foreach ($docs as $h) {
            # prendo solo la parte intera.
            $prefixWithIncrement = $h[Data::ATTRIBUTE_TO_CHECK];
            $incrementString = str_replace($customNumeration, '', $prefixWithIncrement);
            $increment = (int)$incrementString;

            // entro solo se è il primo documento e se non è stato richiesto un doc preciso
            if (!$iniNumber && !$lastIncrementChecked) {
                $lastIncrementChecked = $increment;
                continue;
            }

            if (($increment - $lastIncrementChecked) !== 1) {
                $increment = $lastIncrementChecked + 1;
                if ($iniNumber && !$lastIncrementChecked) {
                    $increment = $iniNumber;
                }
                $newZeroPath = $this->getNewZeroPath($increment);
                $this->queryToRun = array_merge($this->queryToRun, $this->getQueryToUpdate($table, $customNumeration, $increment, $newZeroPath, $h['entity_id']));
            }

            $lastIncrementChecked = $increment;
        }

        return $lastIncrementChecked;
    }

    /**
     * @param $table
     * @param $prefix
     * @param $increment
     * @param $zeroPart
     * @param $entity
     * @return array
     */
    private function getQueryToUpdate($table, $prefix, $increment, $zeroPart, $entity)
    {
        $sequenceValidIncrementIdAsString = (string)$prefix . $zeroPart . $increment;
        $type = (strpos($table, 'invoice') !== false) ? 'Accompagnatoria' : 'Nota di Credito';

        return [
            "UPDATE " . $table . " SET " . Data::ATTRIBUTE_TO_CHECK . " = '" . $sequenceValidIncrementIdAsString . "', " . Data::ATTRIBUTE_TYPE_TO_CHECK . " = '" . $type . "' WHERE entity_id = " . $entity . "",
            "UPDATE " . $table . "_grid SET " . Data::ATTRIBUTE_TO_CHECK . " = '" . $sequenceValidIncrementIdAsString . "' WHERE entity_id = " . $entity . "",
            "UPDATE " . $table . "_comment SET comment = 'Custom fe_increment_id: " . $sequenceValidIncrementIdAsString . "' WHERE parent_id = " . $entity . "",
        ];
    }

    /**
     * @param $increment
     * @return string
     */
    private function getNewZeroPath($increment)
    {
        $str = '';
        for ($i = 0; $i < ($this->getIncrementIdDigitsNumber() - strlen($increment)); $i++) {
            $str .= '0';
        }

        return $str;
    }

    /**
     * @param $table
     * @param $lastIncrementId
     * @return string
     * @throws \Exception
     */
    private function checkSequence($table, $lastIncrementId)
    {
        $sqlCheck = "SELECT * FROM {$table} WHERE sequence_value>={$lastIncrementId}";

        $conn = $this->_connection->getConnection();
        $r = $conn->query(
            $sqlCheck
        );
        $checkData = $r->fetchAll();
        if (count($checkData) > 0) {
            try {
                $r = $conn->query(
                    "DELETE FROM {$table} WHERE sequence_value>={$lastIncrementId}"
                );
            } catch
            (\Exception $e) {
                $conn->rollBack();
                throw $e;
            }
        }

        return "INSERT INTO {$table} (sequence_value) VALUES ($lastIncrementId)";
    }

    /**
     * @param $table
     * @return false|string
     * @throws \Exception
     */
    private function getSequenceCreatedAt($table)
    {
        try {
            $conn = $this->_connection->getConnection();
            $r = $conn->query(
                "SHOW TABLE STATUS WHERE NAME='{$table}'"
            );
            $data = $r->fetchAll();
        } catch (\Exception $e) {
            $conn->rollBack();
            throw $e;
        }

        return date('Y-m-d', strtotime($data[0]['Create_time']));
    }

    /**
     * @param $conn
     * @param $table
     * @param $customTaxVat
     * @param $countryCode
     * @param $sequenceCreatedAt
     * @param $dateIni
     * @param $dateEnd
     * @param bool $isPreTaxRepresentation
     * @param bool $secondIteration
     * @return mixed
     */
    private function getDocuments($conn, $table, $customTaxVat, $countryCode, $sequenceCreatedAt, $dateIni, $dateEnd, $isPreTaxRepresentation = false, $secondIteration = false)
    {
        $singleCountryTaxVat = false;
        $customTaxVatCountries = explode(',', $customTaxVat->getCountries());
        if (count($customTaxVatCountries) == 1) {
            $singleCountryTaxVat = true;
        }

        $preTaxWhere = '';
        if (
            in_array($countryCode, self::OLD_TAX_REPRESENTATION)
            && $table == 'sales_creditmemo'
        ) {
            $preTaxWhere = $isPreTaxRepresentation ?
                " AND o.created_at < '" . self::TAX_REFORM_BREAKDATE . "'" : " AND o.created_at >= '" . self::TAX_REFORM_BREAKDATE . "'";
        }

        $sql = "
            SELECT fe_increment_id,i.entity_id
            FROM {$table} AS i
            LEFT JOIN sales_order AS o ON (i.order_id=o.entity_id)
            LEFT JOIN sales_order_address AS oa ON (o.entity_id=oa.parent_id)
            WHERE oa.country_id='{$customTaxVat->getCountries()}'
            AND oa.address_type='shipping'
            AND i.created_at BETWEEN '{$dateIni} 00:00:00' AND '{$dateEnd} 23:59:59'
            {$preTaxWhere}
            ORDER BY i.created_at ASC
        ";

        if (!$singleCountryTaxVat) {
            $customTaxVatCountriesString = implode("','", $customTaxVatCountries);
            $euCountries = $this->getEuropeanCountries();
            $euCountrisString = implode("','", $euCountries);
            $sql = "
                SELECT fe_increment_id,i.entity_id
                FROM {$table} AS i
                LEFT JOIN sales_order AS o ON (i.order_id=o.entity_id)
                LEFT JOIN sales_order_address AS oa ON (o.entity_id=oa.parent_id)
                WHERE oa.country_id IN ('{$customTaxVatCountriesString}')
                AND (oa.country_id NOT IN ('{$euCountrisString}') OR (oa.country_id IN ('{$euCountrisString}') AND oa.request_invoice=1))
                AND oa.address_type='shipping'
                AND i.created_at BETWEEN '{$dateIni} 00:00:00' AND '{$dateEnd} 23:59:59'
                {$preTaxWhere}
                ORDER BY i.created_at ASC
            ";
        }

        $r = $conn->query(
            $sql
        );
        $tgh = $r->fetchAll();

        if (count($tgh) > 0 || $secondIteration) {
            return $tgh;
        }

        if (!$dateIni = $this->getLastSequenceDocumentDate($conn, $table, $customTaxVat, $countryCode, $dateIni, $sequenceCreatedAt, $isPreTaxRepresentation)) {
            return $tgh;
        }

        return $this->getDocuments($conn, $table, $customTaxVat, $countryCode, $sequenceCreatedAt, $dateIni, $dateEnd, $isPreTaxRepresentation, true);
    }

    /**
     * @param $conn
     * @param $table
     * @param $customTaxVat
     * @param $countryCode
     * @param $dateIni
     * @param $sequenceCreatedAt
     * @param $isPreTaxRepresentation
     * @return bool|false|string
     */
    private function getLastSequenceDocumentDate($conn, $table, $customTaxVat, $countryCode, $dateIni, $sequenceCreatedAt, $isPreTaxRepresentation)
    {
        $singleCountryTaxVat = false;
        $customTaxVatCountries = explode(',', $customTaxVat->getCountries());
        if (count($customTaxVatCountries) == 1) {
            $singleCountryTaxVat = true;
        }

        $preTaxWhere = '';
        if (
            in_array($countryCode, self::OLD_TAX_REPRESENTATION)
            && $table == 'sales_creditmemo'
        ) {
            $preTaxWhere = $isPreTaxRepresentation ?
                " AND o.created_at < '" . self::TAX_REFORM_BREAKDATE . "'" : " AND o.created_at >= '" . self::TAX_REFORM_BREAKDATE . "'";
        }

        $sql = "
            SELECT fe_increment_id,i.entity_id, i.created_at
            FROM {$table} AS i
            LEFT JOIN sales_order AS o ON (i.order_id=o.entity_id)
            LEFT JOIN sales_order_address AS oa ON (o.entity_id=oa.parent_id)
            WHERE oa.country_id='{$customTaxVat->getCountries()}'
            AND oa.address_type='shipping'
            {$preTaxWhere}
            AND fe_increment_id LIKE '{$customTaxVat->getCustomNumeration()}%'
            ORDER BY i.created_at DESC
            LIMIT 1
        ";
        if (!$singleCountryTaxVat) {
            $customTaxVatCountriesString = implode("','", $customTaxVatCountries);
            $euCountries = $this->getEuropeanCountries();
            $euCountrisString = implode("','", $euCountries);
            $sql = "
                SELECT fe_increment_id,i.entity_id, i.created_at
                FROM {$table} AS i
                LEFT JOIN sales_order AS o ON (i.order_id=o.entity_id)
                LEFT JOIN sales_order_address AS oa ON (o.entity_id=oa.parent_id)
                WHERE oa.country_id IN ('{$customTaxVatCountriesString}')
                AND (oa.country_id NOT IN ('{$euCountrisString}') OR (oa.country_id IN ('{$euCountrisString}') AND oa.request_invoice=1))
                AND oa.address_type='shipping'
                AND fe_increment_id LIKE '{$customTaxVat->getCustomNumeration()}%'
                {$preTaxWhere}
                ORDER BY i.created_at DESC
                LIMIT 1
            ";
        }

        $r = $conn->query(
            $sql
        );
        $tgh = $r->fetchAll();

        if (count($tgh) > 0) {
            return date('Y-d-m', strtotime($tgh[0]['created_at']));
        }

        return false;
    }

    /**
     * @return mixed
     */
    private function isSequenceFixEnabled()
    {
        return $this->_scopeConfig->getValue(self::XML_SEQUENCE_FIX_ENABLED_PATH);
    }

    /**
     * @return mixed
     */
    private
    function getIncrementIdDigitsNumber()
    {
        return $this->_scopeConfig->getValue(self::XML_FE_INCREMENT_ID_DIGITS_NUMBER_PATH);
    }

    /**
     * @return array
     */
    private function getEuropeanCountries()
    {
        return explode(',', $this->_scopeConfig->getValue(self::XML_EU_COUNTRIES_PATH));
    }
}
