<?php


namespace Drop\FatturazioneElettronica\Console\Command;


use Magento\Store\Model\ScopeInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Symfony\Component\Console\Input\InputOption;

class TaxReform extends \Symfony\Component\Console\Command\Command
{
    const EU_TAXVAT = [
        "AT" => 20,
        "BE" => 21,
        "BG" => 20,
        "CY" => 19,
        "HR" => 25,
        "DK" => 25,
        "EE" => 20,
        "FI" => 24,
        "FR" => 20,
        "DE" => 19,
        "GR" => 24,
        "IE" => 23,
        "LV" => 21,
        "LT" => 21,
        "LU" => 17,
        "MT" => 18,
        "NL" => 21,
        "PL" => 23,
        "PT" => 23,
        "CZ" => 21,
        "RO" => 19,
        "SK" => 20,
        "SI" => 22,
        "ES" => 21,
        "SE" => 25,
        "HU" => 27,
    ];
    const TAX_CLASS_TAXABLE_GOODS = "Taxable Goods";
    const TAX_CLASS_RETAIL_CUSTOMER = "Retail Customer";
    const XML_PATH_CUSTOM_PREFIX = 'fatturazione_elettronica/general/prefix';
    const XML_PATH_REQUEST_INVOICE_COUNTRIES = 'fatturazione_elettronica/general/req_request_invoice_countries';
    const DROP_ITALIAN_TAXVAT = "01383870431";

    protected $output;
    protected $_appState;
    protected $rule;
    protected $rate;
    protected $productTaxClass;
    protected $customerTaxClass;
    protected $taxvatFactory;
    protected $eventManager;
    protected $scopeConfig;
    protected $configFactory;

    public function __construct(
        \Magento\Framework\App\State $appState,
        \Magento\Tax\Model\Calculation\Rule $rule,
        \Magento\Tax\Model\Calculation\Rate $rate,
        \Magento\Tax\Model\TaxClass\Source\Product $productTaxClass,
        \Magento\Tax\Model\TaxClass\Source\Customer $customerTaxClass,
        \Drop\FatturazioneElettronica\Model\TaxvatNumerationFactory $taxvatFactory,
        \Magento\Framework\Event\Manager $eventManager,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Config\Model\Config\Factory $configFactory
    )
    {
        $this->_appState = $appState;
        $this->rule = $rule;
        $this->rate = $rate;
        $this->productTaxClass = $productTaxClass;
        $this->customerTaxClass = $customerTaxClass;
        $this->taxvatFactory = $taxvatFactory;
        $this->eventManager = $eventManager;
        $this->scopeConfig = $scopeConfig;
        $this->configFactory = $configFactory;
        parent::__construct();
    }

    protected function configure()
    {
        $options = [];
        $this->setName('drop:tax:reform');
        $this->setDescription('creazione/modifica configurazione tasse riforma 1 luglio 2021');
        $this->setDefinition($options);
        parent::configure();
    }

    protected function execute(
        \Symfony\Component\Console\Input\InputInterface $input,
        \Symfony\Component\Console\Output\OutputInterface $output
    )
    {
        $this->output = $output;
        try {
            $this->_appState->setAreaCode('frontend');
        } catch (\Exception $e) {

        }

        /**
         * TODO
         * creo/modifico tutti i tassi per le varie country
         * creo/modifico tutte le rules in base ai tassi inseriti
         * cerco ed elimino le varie combinazioni presenti
         * aggiungo la combinazione rate/rule corretta
         */

        // recupero customer_tax_class_id [Retail Customer]
        $customerTaxClassId = '';
        $customerTaxClasses = $this->customerTaxClass->getAllOptions();
        foreach ($customerTaxClasses as $customerTaxClass) {
            if ($customerTaxClass['label'] != self::TAX_CLASS_RETAIL_CUSTOMER) {
                continue;
            }

            $customerTaxClassId = $customerTaxClass['value'];
            break;
        }
        // recupero product_tax_class_id [Taxable Goods]
        $productTaxClassId = '';
        $productTaxClasses = $this->productTaxClass->getAllOptions();
        foreach ($productTaxClasses as $productTaxClass) {
            if ($productTaxClass['label'] != self::TAX_CLASS_TAXABLE_GOODS) {
                continue;
            }

            $productTaxClassId = $productTaxClass['value'];
            break;
        }

        // recupero prefisso fatture
        $customPrefix = $this->scopeConfig->getValue(self::XML_PATH_CUSTOM_PREFIX, ScopeInterface::SCOPE_STORE);
        $explCustomPrefix = explode('-', $customPrefix);

        foreach (self::EU_TAXVAT as $countryId => $percent) {
            $this->output->writeln("<info>Country {$countryId}</info>");

            $this->output->writeln("<comment>- Set/Update Tax Rate</comment>");
            $this->manageTaxRate($countryId, $percent);

            $this->output->writeln("<comment>- Set/Update Tax Rule</comment>");
            $this->manageTaxRule($countryId, $percent, $customerTaxClassId, $productTaxClassId);

            $this->output->writeln("<comment>- Set/Update Custom Tax Numeration</comment>");
            $this->manageCustomTaxvatNumeration($countryId, $explCustomPrefix[0]);

        }

        $this->output->writeln("<info>Update Drop Module Configuration</info>");
        $this->manageDropConfiguration();
    }

    /**
     * @param $countryId
     * @param $percent
     */
    protected function manageTaxRate($countryId, $percent)
    {
        try {
            if ($taxRate = $this->rate->getCollection()
                ->addFieldToFilter('code', $countryId)
                ->getFirstItem()) {
                // aggiorno
                $taxRate->setRate($percent);
                $taxRate->save();
            } else {
                // creo
                $this->rate->setTaxCountryId($countryId);
                $this->rate->setRegionId(0);
                $this->rate->setTaxPostcode("*");
                $this->rate->setCode($countryId);
                $this->rate->setRate($percent);
                $this->rate->save();
            }
        } catch (\Exception $e) {
            $this->output->writeln("<error>=> ERRORE {$e->getMessage()}</error>");
        }
    }

    /**
     * @param $countryId
     * @param $percent
     * @param $customerTaxClassId
     * @param $productTaxClassId
     */
    protected function manageTaxRule($countryId, $percent, $customerTaxClassId, $productTaxClassId)
    {
        $taxRateId = $this->rate->getCollection()
            ->addFieldToFilter('code', $countryId)
            ->getFirstItem()->getData('tax_calculation_rate_id');

        try {
            // cancello la country dalla regola dov'è attualmente inserita
            $rules = $this->rule->getCollection();
            foreach ($rules as $rule) {
                if (
                    in_array($countryId, $rule->getTaxRatesCodes())
                    && $rule->getCode() != "TAX-{$percent}"
                ) {
                    // cancello la country da questa regola
                    $oldTaxRates = $rule->getTaxRateIds();

                    if(count($oldTaxRates) == 1) {
                        // devo cancellare la regola
                        $rule->delete();
                        break;
                    }

                    unset($oldTaxRates[array_search($taxRateId, $oldTaxRates)]);
                    $rule->setTaxRateIds($oldTaxRates);
                    $rule->save();
                    break;
                }
            }

            // cerco se esiste la regola per quella percentuale
            $rule = $this->rule->getCollection()
                ->addFieldToFilter('code', ['eq' => "TAX-{$percent}"])
                ->getFirstItem();
            if (!count($rule->getData())) {
                // se non esiste la creo associandogli direttamente il taxRate
                $this->rule->setCode("TAX-{$percent}");
                $this->rule->setPriority(0);
                $this->rule->setCustomerTaxClassIds([$customerTaxClassId]);
                $this->rule->setProductTaxClassIds([$productTaxClassId]);
                $this->rule->setTaxRateIds([$taxRateId]);
                $this->rule->save();

                return;
            }

            // se esiste associo alla regola la country
            $oldTaxRates = $rule->getTaxRateIds();
            $rule->setTaxRateIds(array_merge($oldTaxRates, [$taxRateId]));
            $rule->save();
        } catch (\Exception $e) {
            $this->output->writeln("<error>=> ERRORE {$e->getMessage()}</error>");
        }
    }

    /**
     * @param $countryId
     * @param $prefix
     */
    protected function manageCustomTaxvatNumeration($countryId, $prefix)
    {
        try {
            $customTaxvatNumeration = $this->taxvatFactory->create()->getCollection()
                ->addFieldToFilter('countries', ['like' => "%{$countryId}%"])
                ->getFirstItem();
            $customTaxvatNumerationCountries = explode(',', $customTaxvatNumeration->getCountries());
            if (count($customTaxvatNumerationCountries) == 1) {
                $customTaxvatNumeration->setData('custom_taxvat', self::DROP_ITALIAN_TAXVAT);
                $customTaxvatNumeration->save();
                return;
            }

            if (count($customTaxvatNumerationCountries) > 1) {
                unset($customTaxvatNumerationCountries[array_search($countryId, $customTaxvatNumerationCountries)]);
                $customTaxvatNumeration->setData('countries', implode(',', $customTaxvatNumerationCountries));
                $customTaxvatNumeration->save();
                $this->eventManager->dispatch('custom_taxvat_numeration_edit_after', ['data' => $customTaxvatNumeration->getData()]);
            }

            // creo
            $year = date('y');
            $newCustomTaxvatNumeration = $this->taxvatFactory->create();
            $newCustomTaxvatNumerationData = [
                'countries'         => $countryId,
                'custom_taxvat'     => self::DROP_ITALIAN_TAXVAT,
                'custom_numeration' => "{$prefix}-{$countryId}{$year}-",
                'initial_number'    => 1,
            ];
            $newCustomTaxvatNumeration->setData($newCustomTaxvatNumerationData);
            $newCustomTaxvatNumeration->save();
            $customTaxvatNumerationCreate = $this->taxvatFactory->create()->getCollection()
                ->addFieldToFilter('custom_numeration', ['eq' => "{$prefix}-{$countryId}{$year}-"])
                ->getFirstItem();
            $this->eventManager->dispatch('custom_taxvat_numeration_add_after', ['data' => $customTaxvatNumerationCreate->getData()]);
        } catch (\Exception $e) {
            $this->output->writeln("<error>=> ERRORE {$e->getMessage()}</error>");
        }
    }

    /**
     * @throws \Exception
     */
    protected function manageDropConfiguration()
    {
        $reqInvConf = $this->scopeConfig->getValue(self::XML_PATH_REQUEST_INVOICE_COUNTRIES);
        $explReqInvs = explode(',', $reqInvConf);
        foreach (self::EU_TAXVAT as $countryId => $percent) {
            if (in_array($countryId, $explReqInvs)) {
                continue;
            }

            $explReqInvs[] = $countryId;
        }
        asort($explReqInvs);

        $configData = [
            'section' => 'fatturazione_elettronica',
            'website' => null,
            'store'   => null,
            'groups'  => [
                'general' => [
                    'fields' => [
                        'req_request_invoice_countries' => [
                            'value' => $explReqInvs,
                        ],
                    ],
                ],
            ],
        ];
        $configModel = $this->configFactory->create(['data' => $configData]);
        $configModel->save();
    }
}
