<?php

namespace Drop\Email\Model;

use Drop\Email\Helper\VersionCompare;

/**
* Class Message
*/
class Message extends \Magento\Framework\Mail\Message implements \Magento\Framework\Mail\MailMessageInterface
{
    /**
     * @var \Zend\Mime\PartFactory
     */
    private $partFactory;

    /**
     * @var \Zend\Mime\MessageFactory
     */
    private $mimeMessageFactory;

    /**
     * @var \Zend\Mail\Message
     */
    private $zendMessage;

    /**
     * @var mixed
     */
    private $attachment;

    /**
     * @var string
     */
    private $messageType = self::TYPE_TEXT;

    /**
     * Construct
     * @param \Zend\Mime\PartFactory    $partFactory
     * @param \Zend\Mime\MessageFactory $mimeMessageFactory
     * @param string                    $charset
     */
    public function __construct(
        \Zend\Mime\PartFactory $partFactory,
        \Zend\Mime\MessageFactory $mimeMessageFactory,
        $charset = 'utf-8'
    ) {
        $this->partFactory        = $partFactory;
        $this->mimeMessageFactory = $mimeMessageFactory;
        $this->zendMessage        = new \Zend\Mail\Message();

        $this->zendMessage = \Zend\Mail\MessageFactory::getInstance();
        $this->zendMessage->setEncoding($charset);
    }

    /**
     * alias function
     */
    public function createAttachment(
        $content,
        $mimeType = \Zend\Mime\Mime::TYPE_OCTETSTREAM,
        $disposition = \Zend\Mime\Mime::DISPOSITION_ATTACHMENT,
        $encoding = \Zend\Mime\Mime::ENCODING_BASE64,
        $fileName = null
    ) {
        return $this->setBodyAttachment($content, $fileName);
    }

    /**
     * {@inheritdoc}
     */
    public function setBodyAttachment($content, $fileName)
    {
        $attachmentPart = $this->partFactory->create();

        $attachmentPart->setContent($content)
                ->setType(\Zend\Mime\Mime::TYPE_OCTETSTREAM)
                ->setEncoding(\Zend\Mime\Mime::ENCODING_BASE64)
                ->setFileName($fileName)
                ->setDisposition(\Zend\Mime\Mime::DISPOSITION_ATTACHMENT);

        $this->attachment = $attachmentPart;
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function setMessageType($type)
    {
        $this->messageType = $type;
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function setBody($body)
    {
        if (is_string($body) && $this->messageType === \Magento\Framework\Mail\MailMessageInterface::TYPE_HTML) {
            $body = $this->createHtmlMimeFromString($body);
        }

        $attachment = $this->attachment;
        if (isset($attachment)) {
            $body->addPart($attachment);
        }

        $this->zendMessage->setBody($body);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function setSubject($subject)
    {
        $this->zendMessage->setSubject($subject);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getSubject()
    {
        return $this->zendMessage->getSubject();
    }

    /**
     * {@inheritdoc}
     */
    public function getBody()
    {
        return $this->zendMessage->getBody();
    }

    /**
     * {@inheritdoc}
     */
    public function setFrom($fromAddress, $fromName = null)
    {
        $this->setFromAddress($fromAddress, $fromName);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function setFromAddress($fromAddress, $fromName = null)
    {
        $this->zendMessage->setFrom($fromAddress, $fromName);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function addTo($toAddress)
    {
        $this->zendMessage->addTo($toAddress);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function addCc($ccAddress)
    {
        $this->zendMessage->addCc($ccAddress);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function addBcc($bccAddress)
    {
        $this->zendMessage->addBcc($bccAddress);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function setReplyTo($replyToAddress)
    {
        $this->zendMessage->setReplyTo($replyToAddress);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getRawMessage()
    {
        return $this->zendMessage->toString();
    }

    /**
     * Create HTML mime message from the string.
     *
     * @param string $htmlBody
     * @return \Zend\Mime\Message
     */
    private function createHtmlMimeFromString($htmlBody)
    {
        $htmlPart = $this->partFactory->create(['content' => $htmlBody]);
        $htmlPart->setCharset($this->zendMessage->getEncoding());
        $htmlPart->setType(\Zend\Mime\Mime::TYPE_HTML);
        $mimeMessage = $this->mimeMessageFactory->create();
        $mimeMessage->addPart($htmlPart);
        return $mimeMessage;
    }

    /**
     * {@inheritdoc}
     */
    public function setBodyHtml($html)
    {
        $this->setMessageType(self::TYPE_HTML);
        return $this->setBody($html);
    }

    /**
     * {@inheritdoc}
     */
    public function setBodyText($text)
    {
        $this->setMessageType(self::TYPE_TEXT);
        return $this->setBody($text);
    }
}
