<?php

namespace Drop\BetterNewsletter\Model;

use Magento\Framework\App\Request\Http;
use \Magento\Newsletter\Model\Subscriber;
use \Drop\BetterNewsletter\Logger\Logger;

class SubscriberPlugin
{
    /**
     * @var Http
     */
    private $request;
    /**
     * @var \Magento\Customer\Model\Session
     */
    private $customerSession;
    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    private $messageManager;
    /**
     * @var \Magento\Customer\Api\CustomerRepositoryInterface
     */
    private $customerRepositoryInterface;
    /**
     * @var \Drop\BetterNewsletter\Helper\Data
     */
    private $helperData;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var Logger
     */
    private $logger;
    /**
     * @var \Magento\Customer\Model\Session
     */

    /**
     * NewsletterPlugin constructor.
     * @param Http $request
     * @param \Magento\Customer\Model\Session $customerSession
     * @param \Magento\Customer\Model\CustomerFactory $customerFactory
     * @param \Magento\Framework\Message\ManagerInterface $messageManager
     */
    public function __construct(
        Http $request,
        \Magento\Customer\Model\Session $customerSession,
        \Magento\Customer\Api\CustomerRepositoryInterface $customerRepositoryInterface,
        \Magento\Framework\Message\ManagerInterface $messageManager,
        \Drop\BetterNewsletter\Helper\Data $helperData,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        Logger $logger
    ) {
        $this->request = $request;
        $this->customerSession = $customerSession;
        $this->messageManager = $messageManager;
        $this->customerRepositoryInterface = $customerRepositoryInterface;
        $this->helperData = $helperData;
        $this->storeManager = $storeManager;
        $this->logger = $logger;
    }

    public function beforeSave(Subscriber $subscriber) {

        if (empty($this->request->getPostValue())) {
            return $subscriber;
        }

        if(!$params = $this->validatedParams($this->request)) {
            return $subscriber;
        }

        //Prevent endless loop
        if($this->customerSession->getBetterNewsletterSubscriberPluginBeforeSaveExecuted()) {
            //Reset flag for upcoming saves
            $this->customerSession->setBetterNewsletterSubscriberPluginBeforeSaveExecuted(false);
            return $subscriber;
        }
        $this->customerSession->setBetterNewsletterSubscriberPluginBeforeSaveExecuted(true);

        //Add subscriber custom data
        $subscriber->addData($this->getFormattedSubscriberData($params));

        //Sync Customer missing info
        $customer = $this->getIfCustomerAlreadyExist($params['email']);
        if(!$customer || !$customer->getId()) {
            return $subscriber;
        }

        $this->setMissingCustomerData($customer, $subscriber);

        return $subscriber;
    }

    private function validatedParams($request)
    {
        if (false === \strpos($request->getParam('email'), '@')) {
            return false;
        }

        return $request->getParams();
    }

    /**
     * Check if customer is logged or exist by email
     * @param $email
     * @return bool|\Magento\Customer\Model\Customer
     */
    public function getIfCustomerAlreadyExist($email) {
        try {
            $loggedCustomer = $this->customerSession->getCustomer();
            $nonLoggedCustomer = $this->customerRepositoryInterface->get($email, $this->storeManager->getStore()->getWebsiteId());
        } catch (\Exception $e) {
            return false;
        }

        if($loggedCustomer->getId()) {
            return $loggedCustomer;
        }
        if($nonLoggedCustomer->getId()) {
            return $nonLoggedCustomer;
        }

        return false;
    }

    /**
     * Se il customer esiste provo ad aggiornare i suoi campi con le nuove informazione dell'iscrizione alla newsletter.
     * Comanda sempre il customer quindi:
     *      non aggiorno i campi obbligatori come firstname, lastname. Prendo sempre quelli del customer se esiste
     *      aggiorno i campi opzionali con i dati del subscriber solo se quelli del customer sono vuoti, altrimenti quelli hanno priorità
     * Aggiungo che non ci interessa il contrario:
     *      se il customer prima si iscrive alla newsletter e poi diventa customer con meno dati inseriti, non vado ad aggiungere i dati della newsletter al customer perchè si troverebbe dati che non ricorda di aver inserito
     * @param $customer
     * @param $subscriber
     */
    public function setMissingCustomerData($customer, $subscriber) {
        $isChanged = false;

        if(empty($customer->getDob()) && !empty($subscriber->getSubscriberDob())) {
            $isChanged = true;
            $customer->setDob($this->helperData->getDobDateFormatted($subscriber->getSubscriberDob()));
        }
        if(empty($customer->getGender()) && !empty($subscriber->getSubscriberGender())) {
            $isChanged = true;
            $customer->setGender($subscriber->getSubscriberGender());
        }
        if(($customer->getCustomAttribute('is_profiling_accepted') && !$customer->getCustomAttribute('is_profiling_accepted')->getValue()) && $subscriber->getIsProfilingAccepted()) {
            $isChanged = true;
            $customer->setCustomAttribute('is_profiling_accepted', $subscriber->getIsProfilingAccepted());
        }

        if(!$isChanged) {
            return false;
        }

        try {
            $this->customerRepositoryInterface->save($customer);
        } catch (\Exception $e) {
            $this->logger->info($e->getMessage());
            $this->messageManager->addError($e->getMessage());
        }
    }

    public function getFormattedSubscriberData($data) {
        if(isset($data['email'])) {
            $data['subscriber_email'] = $data['email'];
        }
        if(isset($data['customer']['email'])) {
            $data['subscriber_email'] = $data['customer']['email'];
        }

        if(isset($data['name'])) {
            $data['subscriber_firstname'] = $data['name'];
        }
        if(isset($data['firstname'])) {
            $data['subscriber_firstname'] = $data['firstname'];
        }
        if(isset($data['customer']['firstname'])) {
            $data['subscriber_firstname'] = $data['customer']['firstname'];
        }
        if(isset($data['lastname'])) {
            $data['subscriber_lastname'] = $data['lastname'];
        }
        if(isset($data['customer']['lastname'])) {
            $data['subscriber_lastname'] = $data['customer']['lastname'];
        }

        if(isset($data['dob'])) {
            $data['subscriber_dob'] = $data['dob'];
        }
        if(isset($data['customer']['dob'])) {
            $data['subscriber_dob'] = $data['customer']['dob'];
        }
        if(isset($data['gender'])) {
            $data['subscriber_gender'] = $data['gender'];
        }
        if(isset($data['customer']['gender'])) {
            $data['subscriber_gender'] = $data['customer']['gender'];
        }

        return $data;
    }

}
