<?php
namespace Drop\BetterNewsletter\Model;

use Magento\Customer\Model\Customer;
use Magento\Customer\Model\Session;
use Magento\Framework\App\Request\Http;

class CustomerPlugin
{
    /**
     * @var Http
     */
    private $request;
    /**
     * @var \Magento\Newsletter\Model\Subscriber
     */
    private $subscriber;
    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    private $messageManager;
    /**
     * @var Session
     */
    private $customerSession;

    /**
     * CustomerPlugin constructor.
     * @param Http $request
     * @param \Magento\Newsletter\Model\Subscriber $subscriber
     * @param \Magento\Framework\Message\ManagerInterface $messageManager
     */
    public function __construct(
        Http $request,
        Session $customerSession,
        \Magento\Newsletter\Model\Subscriber $subscriber,
        \Magento\Framework\Message\ManagerInterface $messageManager
    ) {
        $this->request = $request;
        $this->customerSession = $customerSession;
        $this->subscriber = $subscriber;
        $this->messageManager = $messageManager;
    }

    /**
     * This is only for customer information update
     * On account creation this will exit because SubscriberPlugin will be used
     * @param Customer $customer
     * @return Customer
     */
    public function beforeSave(Customer $customer)
    {
        if (empty($this->request->getPostValue())) {
            return $customer;
        }

        //Prevent endless loop
        if($this->customerSession->getBetterNewsletterCustomerPluginBeforeSaveExecuted()) {
            //Reset flag for upcoming saves
            $this->customerSession->setBetterNewsletterCustomerPluginBeforeSaveExecuted(false);
            return $customer;
        }
        $this->customerSession->setBetterNewsletterCustomerPluginBeforeSaveExecuted(true);

        $customerId = $customer->getId();
        if(!$customerId) {
            return $customer;
        }

        $subscriber = $this->subscriber->loadByCustomerId($customerId);
        if(!$subscriber || !$subscriber->isSubscribed()) {
            return $customer;
        }

        $subscriberData = $this->getFormattedSubscriberData($this->request->getParams());
        if(empty($subscriberData)) {
            return $customer;
        }

        //Add subscriber custom data
        $subscriber->addData($subscriberData);

        try {
            //Trigger newsletter plugin beforeSave to sync newsletter content
            $subscriber->save();
        } catch (\Exception $e) {
            $this->messageManager->addError($e->getMessage());
        }

        return $customer;
    }

    public function getFormattedSubscriberData($data) {
        if(isset($data['firstname'])) {
            $data['subscriber_firstname'] = $data['firstname'];
        }
        if(isset($data['customer']['firstname'])) {
            $data['subscriber_firstname'] = $data['customer']['firstname'];
        }
        if(isset($data['lastname'])) {
            $data['subscriber_lastname'] = $data['lastname'];
        }
        if(isset($data['customer']['lastname'])) {
            $data['subscriber_lastname'] = $data['customer']['lastname'];
        }

        if(isset($data['dob'])) {
            $data['subscriber_dob'] = $data['dob'];
        }
        if(isset($data['customer']['dob'])) {
            $data['subscriber_dob'] = $data['customer']['dob'];
        }
        if(isset($data['gender'])) {
            $data['subscriber_gender'] = $data['gender'];
        }
        if(isset($data['customer']['gender'])) {
            $data['subscriber_gender'] = $data['customer']['gender'];
        }
        
        return $data;
    }

}
