<?php
/**
 * CachePlugin
 *
 * @copyright Copyright © 2020 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Bettercatalogimageresize\Plugin\Product\Image;

use Magento\Catalog\Helper\Image as ImageHelper;
use Magento\Framework\App\Area;
use Magento\Framework\View\ConfigInterface;
use Magento\Theme\Model\ResourceModel\Theme\Collection as ThemeCollection;
use Psr\Log\LoggerInterface;
use Magento\Theme\Model\Config\Customization as ThemeCustomizationConfig;
use Magento\Catalog\Model\Product;
use Magento\Eav\Model\Config;
use Magento\Eav\Model\Entity\Attribute;
use Magento\Framework\ObjectManagerInterface;

class CachePlugin extends \Magento\Catalog\Model\Product\Image\Cache
{
    /**
     * @var \Magento\Theme\Model\Config\Customization
     */
    private $themeCustomizationConfig;
    /**
     * @var \Psr\Log\LoggerInterface|null
     */
    private $logger;
    /**
     * @var \Magento\Eav\Model\Config
     */
    private $eavConfig;
    /**
     * @var \Magento\Eav\Model\Entity\Attribute
     */
    private $attribute;
    /**
     * @var \Magento\Framework\ObjectManagerInterface
     */
    private $objectManager;

    /**
     * CachePlugin constructor.
     * @param \Magento\Framework\View\ConfigInterface $viewConfig
     * @param \Magento\Theme\Model\ResourceModel\Theme\Collection $themeCollection
     * @param \Magento\Catalog\Helper\Image $imageHelper
     * @param \Psr\Log\LoggerInterface|null $logger
     * @param \Magento\Theme\Model\Config\Customization $themeCustomizationConfig
     * @param \Magento\Eav\Model\Config $eavConfig
     * @param \Magento\Eav\Model\Entity\Attribute $attribute
     * @param \Magento\Framework\ObjectManagerInterface $objectManager
     */
    public function __construct(
        ConfigInterface $viewConfig,
        ThemeCollection $themeCollection,
        ImageHelper $imageHelper,
        LoggerInterface $logger = null,
        ThemeCustomizationConfig $themeCustomizationConfig,
        Config $eavConfig,
        Attribute $attribute,
        ObjectManagerInterface $objectManager
    )
    {
        $this->logger = $logger;
        $this->themeCustomizationConfig = $themeCustomizationConfig;
        $this->eavConfig = $eavConfig;
        $this->attribute = $attribute;
        $this->objectManager = $objectManager;
        parent::__construct($viewConfig, $themeCollection, $imageHelper, $logger);
    }

    protected function getData()
    {
        if (!$this->data) {
            $themes = $this->getThemesInUse();

            /** @var \Magento\Theme\Model\Theme $theme */
            foreach ($themes as $theme) {
                $config = $this->viewConfig->getViewConfig([
                    'area' => Area::AREA_FRONTEND,
                    'themeModel' => $theme,
                ]);
                $images = $config->getMediaEntities('Magento_Catalog', ImageHelper::MEDIA_TYPE_CONFIG_NODE);
                foreach ($images as $imageId => $imageData) {
                    $this->data[$theme->getCode() . $imageId] = array_merge(['id' => $imageId], $imageData);
                }
            }
        }
        return $this->data;
    }

    /**
     * Resize product images and save results to image cache.
     *
     * @param Product $product
     *
     * @return $this
     * @throws \Exception
     */
    public function generate(Product $product)
    {
        $isException = false;
        $galleryImages = $product->getMediaGalleryImages();
        if ($galleryImages) {
            foreach ($galleryImages as $image) {
                foreach ($this->getData() as $imageData) {
                    try {
                        $this->processImageData($product, $imageData, $image->getFile());
                    } catch (\Exception $e) {
                        $isException = true;
                        $this->logger->error($e->getMessage());
                        $this->logger->error(__('The image could not be resized: ') . $image->getPath());
                    }
                }
            }
        }

        if ($isException === true) {
            throw new \Magento\Framework\Exception\RuntimeException(
                __('Some images could not be resized. See log file for details.')
            );
        }

        return $this;
    }

    /**
     * Get themes in use
     *
     * @return array
     */
    protected function getThemesInUse()
    {
        $themesInUse = [];

        $registeredThemes = $this->themeCollection->loadRegisteredThemes();
        $storesByThemes   = $this->themeCustomizationConfig->getStoresByThemes();
        foreach ($registeredThemes as $registeredTheme) {
            if (array_key_exists($registeredTheme->getThemeId(), $storesByThemes)) {
                $themesInUse[] = $registeredTheme;
            }
        }

        $resource = $this->objectManager->get(\Magento\Framework\App\ResourceConnection::class);

        $catalogProductEntityId = $this->eavConfig->getEntityType(\Magento\Catalog\Model\Product::ENTITY)->getId();
        $productCustomDesignAttributeId = $this->attribute->loadByCode($catalogProductEntityId, 'custom_design')->getId();

        $productSql = $resource->getConnection()
            ->select()
            ->from(
                ['eav' => $resource->getTableName('catalog_product_entity_varchar')],
                ['value']
            )
            ->where('eav.attribute_id = ?', $productCustomDesignAttributeId)
            ->where('eav.value > 0')
            ->group('value');

        $productThemeIds = $resource->getConnection()->fetchCol($productSql);
        foreach ($productThemeIds as $productThemeId) {
            if (array_key_exists($productThemeId, $storesByThemes) && !array_key_exists($productThemeId, $themesInUse) ) {
                $themesInUse[] = $this->themeCollection->load($productThemeId);
            }
        }

        $catalogCategoryEntityId = $this->eavConfig->getEntityType(\Magento\Catalog\Model\Category::ENTITY)->getId();
        $categoryCustomDesignAttributeId = $this->attribute->loadByCode($catalogCategoryEntityId, 'custom_design')->getId();

        $categorySql = $resource->getConnection()
            ->select()
            ->from(
                ['eav' => $resource->getTableName('catalog_category_entity_varchar')],
                ['value']
            )
            ->where('eav.attribute_id = ?', $categoryCustomDesignAttributeId)
            ->where('eav.value > 0')
            ->group('value');

        $categoryThemeIds = $resource->getConnection()->fetchCol($categorySql);
        foreach ($categoryThemeIds as $categoryThemeId) {
            if (array_key_exists($categoryThemeId, $storesByThemes) && !array_key_exists($categoryThemeId, $themesInUse) ) {
                $themesInUse[] = $this->themeCollection->load($categoryThemeId);
            }
        }

        $pageSql = $resource->getConnection()
            ->select()
            ->from(
                ['page' => $resource->getTableName('cms_page')],
                ['custom_theme']
            )
            ->where('custom_theme > 0')
            ->group('custom_theme');

        $pageThemeIds = $resource->getConnection()->fetchCol($pageSql);
        foreach ($pageThemeIds as $pageThemeId) {
            if (array_key_exists($pageThemeId, $storesByThemes) && !array_key_exists($pageThemeId, $themesInUse) ) {
                $themesInUse[] = $this->themeCollection->load($pageThemeId);
            }
        }

        return $themesInUse;
    }

}
