<?php
/**
 * ImagesResizeCommand
 *
 * @copyright Copyright © 2020 Drop. All rights reserved.
 * @author    c.pieroni@drop.it
 */

namespace Drop\Bettercatalogimageresize\Plugin\Console\Command;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Image\CacheFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Catalog\Model\ResourceModel\Product\Image as ProductImage;
use Magento\Framework\App\Area;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\App\State;
use Magento\Catalog\Helper\Image as ImageHelper;
use Magento\Framework\Console\Cli;
use Magento\Theme\Model\Config\Customization as ThemeCustomizationConfig;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Framework\View\ConfigInterface as ViewConfig;
use Magento\Theme\Model\ResourceModel\Theme\Collection as ThemeCollection;
//use Magento\Catalog\Model\Product\Image;
use Drop\Bettercatalogimageresize\Plugin\Product\ImagePlugin as Image;
//use Magento\Catalog\Model\Product\Image as ProductImageFactory;
use \Drop\Bettercatalogimageresize\Plugin\Product\ImagePluginFactory as ProductImageFactory;
use Symfony\Component\Console\Helper\ProgressBar;
use Magento\Eav\Model\Config;
use Magento\Eav\Model\Entity\Attribute;
use Magento\Framework\ObjectManagerInterface;

class ImagesResizeCommand  extends Command // extends \Magento\Catalog\Console\Command\ImagesResizeCommand
{
    /**
     * @var State
     */
    protected $appState;

    /**
     * @deprecated
     * @var CollectionFactory
     */
    protected $productCollectionFactory;

    /**
     * @deprecated
     * @var ProductRepositoryInterface
     */
    protected $productRepository;

    /**
     * @deprecated
     * @var CacheFactory
     */
    protected $imageCacheFactory;

    /**
     * @var ProductImage
     */
    private $productImage;

    /**
     * @var ViewConfig
     */
    private $viewConfig;

    /**
     * @var ThemeCollection
     */
    private $themeCollection;

    /**
     * @var ProductImageFactory
     */
    private $productImageFactory;
    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;
    /**
     * @var \Magento\Theme\Model\Config\Customization
     */
    private $themeCustomizationConfig;
    /**
     * @var \Magento\Eav\Model\Config
     */
    private $eavConfig;
    /**
     * @var \Magento\Eav\Model\Entity\Attribute
     */
    private $attribute;
    /**
     * @var \Magento\Framework\ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @param State $appState
     * @param CollectionFactory $productCollectionFactory
     * @param ProductRepositoryInterface $productRepository
     * @param CacheFactory $imageCacheFactory
     * @param ProductImage $productImage
     * @param ViewConfig $viewConfig
     * @param ThemeCollection $themeCollection
     * @param ProductImageFactory $productImageFactory
     * @param \Magento\Theme\Model\Config\Customization $themeCustomizationConfig
     * @param \Magento\Eav\Model\Config $eavConfig
     * @param \Magento\Eav\Model\Entity\Attribute $attribute
     * @param \Magento\Framework\ObjectManagerInterface $objectManager
     * @param \Psr\Log\LoggerInterface $logger
     */
    public function __construct(
        State $appState,
        CollectionFactory $productCollectionFactory,
        ProductRepositoryInterface $productRepository,
        CacheFactory $imageCacheFactory,
        ProductImage $productImage = null,
        ViewConfig $viewConfig = null,
        ThemeCollection $themeCollection = null,
        ProductImageFactory $productImageFactory = null,
        ThemeCustomizationConfig $themeCustomizationConfig,
        Config $eavConfig,
        Attribute $attribute,
        ObjectManagerInterface $objectManager,
        \Psr\Log\LoggerInterface $logger
    ) {
        $this->appState = $appState;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->productRepository = $productRepository;
        $this->imageCacheFactory = $imageCacheFactory;
        $this->productImage = $productImage ?: ObjectManager::getInstance()->get(ProductImage::class);
        $this->viewConfig = $viewConfig ?: ObjectManager::getInstance()->get(ViewConfig::class);
        $this->themeCollection = $themeCollection ?: ObjectManager::getInstance()->get(ThemeCollection::class);
        $this->themeCustomizationConfig = $themeCustomizationConfig;
        $this->productImageFactory = $productImageFactory
            ?: ObjectManager::getInstance()->get(ProductImageFactory::class);
        $this->logger = $logger;
        $this->eavConfig = $eavConfig;
        $this->attribute = $attribute;
        $this->objectManager = $objectManager;
        parent::__construct();
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this->setName('catalog:images:resize')
            ->setDescription('Creates resized product images');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        try {
            $this->appState->setAreaCode(Area::AREA_GLOBAL);
        } catch (\Exception $e) {}

        try {
            $count = $this->productImage->getCountAllProductImages();
            if (!$count) {
                $output->writeln("<info>No product images to resize</info>");
                return Cli::RETURN_SUCCESS;
            }

            $productImages = $this->productImage->getAllProductImages();
            $themes = $this->getThemesInUse();
            $viewImages = $this->getViewImages($themes->getItems());

            $viewImageCount = count($viewImages);
            $progress = new ProgressBar($output, $count);
            $progress->setFormat(
                "%current%($viewImageCount views)/%max% [%bar%] %percent:3s%% %elapsed% %memory:6s% \t| <info>%message%</info>"
            );

            if ($output->getVerbosity() !== OutputInterface::VERBOSITY_NORMAL) {
                $progress->setOverwrite(false);
            }

            foreach ($productImages as $image) {
                $originalImageName = $image['filepath'];
                foreach ($viewImages as $viewImage) {
                    /** @var \Drop\Bettercatalogimageresize\Plugin\Product\ImagePlugin $image */
                    $image = $this->makeImage($originalImageName, $viewImage);
                    if(file_exists($image->getPath())) {
                        continue;
                    }

                    if(empty($image->getBaseFile())) {
                        $this->logger->error('Cannot find original image file for ' . $originalImageName);
                        $output->write("\n");
                        $output->writeln("<error>Cannot find original image file for " . $originalImageName . "</error>");
                        continue;
                    }

                    $image->resize();
                    $image->saveFile();
                }

                $progress->setMessage($originalImageName);
                $progress->advance();
            }
        } catch (\Exception $e) {
            $output->writeln("<error>{$e->getMessage()}</error>");
            // we must have an exit code higher than zero to indicate something was wrong
            return Cli::RETURN_FAILURE;
        }

        $output->write("\n");
        $output->writeln("<info>Product images resized successfully.</info>");
        return 0;
    }

    /**
     * Get only used themes
     *
     * @return \Magento\Theme\Model\ResourceModel\Theme\Collection
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function getThemesInUse()
    {
        $themesIdsInUse = [];
        $registeredThemes = $this->loadRegisteredThemes();
        $storesByThemes   = $this->themeCustomizationConfig->getStoresByThemes();
        foreach ($registeredThemes as $registeredTheme) {
            if (array_key_exists($registeredTheme->getThemeId(), $storesByThemes) || $registeredTheme->getCode() == 'Magento/backend') {
                $themesIdsInUse[] = $registeredTheme->getThemeId();
            }
        }

        $resource = $this->objectManager->get(\Magento\Framework\App\ResourceConnection::class);

        $catalogProductEntityId = $this->eavConfig->getEntityType(\Magento\Catalog\Model\Product::ENTITY)->getId();
        $productCustomDesignAttributeId = $this->attribute->loadByCode($catalogProductEntityId, 'custom_design')->getId();

        $productSql = $resource->getConnection()
            ->select()
            ->from(
                ['eav' => $resource->getTableName('catalog_product_entity_varchar')],
                ['value']
            )
            ->where('eav.attribute_id = ?', $productCustomDesignAttributeId)
            ->where('eav.value > 0')
            ->group('value');

        $productThemeIds = $resource->getConnection()->fetchCol($productSql);
        foreach ($productThemeIds as $productThemeId) {
            if (array_key_exists($productThemeId, $storesByThemes) && !in_array($productThemeId, $themesIdsInUse)) {
                $themesIdsInUse[] = $productThemeId;
            }
        }

        $catalogCategoryEntityId = $this->eavConfig->getEntityType(\Magento\Catalog\Model\Category::ENTITY)->getId();
        $categoryCustomDesignAttributeId = $this->attribute->loadByCode($catalogCategoryEntityId, 'custom_design')->getId();

        $categorySql = $resource->getConnection()
            ->select()
            ->from(
                ['eav' => $resource->getTableName('catalog_category_entity_varchar')],
                ['value']
            )
            ->where('eav.attribute_id = ?', $categoryCustomDesignAttributeId)
            ->where('eav.value > 0')
            ->group('value');

        $categoryThemeIds = $resource->getConnection()->fetchCol($categorySql);
        foreach ($categoryThemeIds as $categoryThemeId) {
            if (array_key_exists($categoryThemeId, $storesByThemes)
                && !in_array($categoryThemeId, $themesIdsInUse) ) {
                $themesIdsInUse[] = $categoryThemeId;
            }
        }

        $pageSql = $resource->getConnection()
            ->select()
            ->from(
                ['page' => $resource->getTableName('cms_page')],
                ['custom_theme']
            )
            ->where('custom_theme > 0')
            ->group('custom_theme');

        $pageThemeIds = $resource->getConnection()->fetchCol($pageSql);
        foreach ($pageThemeIds as $pageThemeId) {
            if (array_key_exists($pageThemeId, $storesByThemes)
                && !in_array($pageThemeId, $themesIdsInUse) ) {
                $themesIdsInUse[] = $pageThemeId;
            }
        }
        foreach ($registeredThemes as $registeredTheme) {
            if (!in_array($registeredTheme->getThemeId(), $themesIdsInUse)) {
                $registeredThemes->removeItemByKey($registeredTheme->getThemeId());
            }
        }

        return $registeredThemes;
    }

    public function loadRegisteredThemes()
    {
        $this->themeCollection->clear();
        return $this->themeCollection->setOrder('theme_title', \Magento\Framework\Data\Collection::SORT_ORDER_ASC)
            ->filterVisibleThemes();
    }

    /**
     * Get view images data from themes
     * @param array $themes
     * @return array
     */
    private function getViewImages(array $themes): array
    {
        $viewImages = [];
        foreach ($themes as $theme) {
            $config = $this->viewConfig->getViewConfig([
                'area' => Area::AREA_FRONTEND,
                'themeModel' => $theme,
            ]);
            $images = $config->getMediaEntities('Magento_Catalog', ImageHelper::MEDIA_TYPE_CONFIG_NODE);
            foreach ($images as $imageId => $imageData) {
                $uniqIndex = $this->getUniqueImageIndex($imageData);
                $imageData['id'] = $imageId;
                $viewImages[$uniqIndex] = $imageData;
            }
        }
        return $viewImages;
    }

    /**
     * Make image
     * @param string $originalImagePath
     * @param array $imageParams
     * @return Image
     */
    private function makeImage(string $originalImagePath, array $imageParams): Image
    {
        /** @var \Drop\Bettercatalogimageresize\Plugin\Product\ImagePlugin $image */
        $image = $this->productImageFactory->create();

        if (isset($imageParams['height'])) {
            $image->setHeight($imageParams['height']);
        }
        if (isset($imageParams['width'])) {
            $image->setWidth($imageParams['width']);
        }
        if (isset($imageParams['aspect_ratio'])) {
            $image->setKeepAspectRatio($imageParams['aspect_ratio']);
        }
        if (isset($imageParams['frame'])) {
            $image->setKeepFrame($imageParams['frame']);
        }
        if (isset($imageParams['transparency'])) {
            $image->setKeepTransparency($imageParams['transparency']);
        }
        if (isset($imageParams['constrain'])) {
            $image->setConstrainOnly($imageParams['constrain']);
        }
        if (isset($imageParams['background'])) {
            $image->setBackgroundColor($imageParams['background']);
        }

        $image->setDestinationSubdir($imageParams['type']);
        $image->setBaseFile($originalImagePath);
        return $image;
    }

    /**
     * Get unique image index
     * @param array $imageData
     * @return string
     */
    private function getUniqueImageIndex(array $imageData): string
    {
        ksort($imageData);
        unset($imageData['type']);
        return md5(json_encode($imageData));
    }

}
