<?php

namespace Cooder\Sia\Model\Webservice\TransactionList;

/**
 * Oggetto per gestione della richiesta per il recupero delle informazioni
 * sulle transazioni
 * 
 * @author Devid Marcantoni <devid@cooder.it>
 */
class Request
{
    
    const RESPONSE_SUCCESS_CODE = '00';
    const RESPONSE_EMPTY_CODE = '05';
    
    /**
     * Date Object
     *
     * @var \Magento\Framework\Stdlib\DateTime\TimezoneInterface
     */
    protected $_date;
    
    /**
     * Oggetto per le comunicazioni cURL
     *
     * @var \Magento\Framework\HTTP\Client\Curl
     */
    protected $_curl;
    
    /**
     * Xml Parser
     *
     * @var \Magento\Framework\Xml\Parser
     */
    protected $_xmlParser;
    
    /**
     * Oggetto per la memorizzazione della response
     *
     * @var \Cooder\Sia\Model\Webservice\TransactionList\Response
     */
    protected $_response;
    
    /**
     * Logger del modulo
     *
     * @var \Cooder\Sia\Logger\Logger
     */
    protected $_baseLogger;
    
    /**
     * Definisce se si è in modalita debug
     *
     * @var boolean
     */
    protected $_debug = false;
    
    /**
     * Url del webservice per la creazione del link
     * per il pagamento
     *
     * @var string
     */
    protected $_requestUrl;
    
    /**
     * Chiave per la costruzione del
     * codice MAC
     *
     * @var string
     */
    protected $_macRequestKey;
    
    /**
     * Metodologia di criptazione da utilizzare
     * per la creazione del codice MAC
     *
     * @var string
     */
    protected $_hash = \Cooder\Sia\Model\Config\Source\Hash::HASH_SHA1;
    
    /**
     * Transaction requested: to be filled in with "LISTOPERATION"
     *
     * @var string
     */
    protected $_operation = 'LISTOPERATION';
    
    /**
     * Local timestamp of the yyyy-MM-ddTHH:mm:ss.SSS type
     *
     * @var string
     */
    protected $_timestamp;
    
    /**
     * Identifier of the merchant’s store assigned by SIA [MID]
     *
     * @var string
     */
    protected $_shopId;
    
    /**
     * It indicates the person who has requested the transaction
     *
     * @var string
     */
    protected $_operatorId;
    
    /**
     * Unique identifier of the request managed by the merchant.
     * It can be used to retrieve information on the request made also in the case of no  response.
     * The  first  8  digits  must  be  in  the  yyyyMMdd  format including the request date.
     *
     * @var string
     */
    protected $_reqRefNum;
    
    /**
     * Date of start of period, in yyyy-MM-dd format
     * 
     * @var string
     */
    protected $_startDate;
    
    /**
     * Date of end of period, in yyyy-MM-dd format
     *
     * @var string
     */
    protected $_endDate;
    
    /**
     * Release of APIs: to be filled in with “02”
     *
     * @var string
     */
    protected $_release = '02';
    
    /**
     * The field OPTIONS permits to activate various additional options for the ongoing message.
     *
     * @var string
     */
    protected $_options;
    
    /**
     * Type of transaction to be extracted. 
     * The possible values are: 01 02 03 04, which make reference to the field <SrcType> of the element <Operation>. 
     * 
     * @var string
     */
    protected $_srcType;
    
    /**
     * The resulting list includes only the operations with the optional description 
     * 
     * @var string
     */
    protected $_opDescr;
    
    /**
     * Request  signature  field:  it  makes  the  data  sent unchangeable  by third parties
     *
     * @var string
     */
    protected $_mac;
    
    /**
     * Costruttore
     *
     * @param \Magento\Framework\Stdlib\DateTime\TimezoneInterface $date
     * @param \Magento\Framework\HTTP\Client\Curl $curl
     * @param \Magento\Framework\Xml\Parser $xmlParser
     * @param \Cooder\Sia\Model\Webservice\TransactionList\Response $response
     * @param \Cooder\Sia\Logger\Logger $baseLogger
     */
    public function __construct(
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $date,
        \Magento\Framework\HTTP\Client\Curl $curl,
        \Magento\Framework\Xml\Parser $xmlParser,
        \Cooder\Sia\Model\Webservice\TransactionList\Response $response,
        \Cooder\Sia\Logger\Logger $baseLogger
    )
    {
        $this->_date =  $date;
        $this->_curl = $curl;
        $this->_xmlParser = $xmlParser;
        $this->_response = $response;
        $this->_baseLogger = $baseLogger;
    }
    
    /**
     * Imposta se si è in modalita debug
     *
     * @param boolean $value
     */
    public function setDebug($value)
    {
        $this->_debug = $value;
    }
    
    /**
     * Imposta la url del webservice per la richiesta
     * del link di pagamento
     *
     * @param string $value
     */
    public function setRequestUrl($value)
    {
        $this->_requestUrl = $value;
    }
    
    /**
     * Ritorna il campo 'RequestUrl'
     *
     * @return string
     */
    public function getRequestUrl()
    {
        return $this->_requestUrl;
    }
    
    /**
     * Imposta la chiave per la creazione del codice
     * MAC
     *
     * @param string $value
     */
    public function setMacRequestKey($value)
    {
        $this->_macRequestKey = $value;
    }
    
    /**
     * Ritorna il campo 'MacRequestKey'
     *
     * @return string
     */
    public function getMacRequestKey()
    {
        return $this->_macRequestKey;
    }
    
    /**
     * Imposta la tecnologia di criptazione
     * per la generazione del codice MAC
     *
     * @param string $value
     */
    public function setHash($value)
    {
        $this->_hash = $value;
    }
    
    /**
     * Ritorna il campo 'Hash'
     *
     * @return string
     */
    public function getHash()
    {
        return $this->_hash;
    }
    
    /**
     * Set value 'Operation'
     *
     * @param string $value
     */
    public function setOperation($value)
    {
        $this->_operation = $value;
    }
    
    /**
     * Ritorna il campo 'Operation'
     *
     * @return string
     */
    public function getOperation()
    {
        return $this->_operation;
    }
    
    /**
     * Set value 'Timestamp'
     *
     * @param string $value
     */
    public function setTimestamp($value)
    {
        $this->_timestamp = $value;
    }
    
    /**
     * Ritorna il campo 'Timestamp'
     *
     * @return string
     */
    public function getTimestamp()
    {
        return $this->_timestamp;
    }
    
    /**
     * Set value 'ShopId'
     *
     * @param string $value
     */
    public function setShopId($value)
    {
        $this->_shopId = $value;
    }
    
    /**
     * Ritorna il campo 'ShopId'
     *
     * @return string
     */
    public function getShopId()
    {
        return $this->_shopId;
    }
    
    /**
     * Set value 'OperatorId'
     *
     * @param string $value
     */
    public function setOperatorId($value)
    {
        $this->_operatorId = $value;
    }
    
    /**
     * Ritorna il campo 'OperatorId'
     *
     * @return string
     */
    public function getOperatorId()
    {
        return $this->_operatorId;
    }
    
    /**
     * Set value 'ReqRefNum'
     *
     * @param string $value
     */
    public function setReqRefNum($value)
    {
        $this->_reqRefNum = $value;
    }
    
    /**
     * Ritorna il campo 'ReqRefNum'
     *
     * @return string
     */
    public function getReqRefNum()
    {
        return $this->_reqRefNum;
    }
    
    /**
     * Set value 'StartDate'
     *
     * @param string $value
     */
    public function setStartDate($value)
    {
        $this->_startDate = $value;
    }
    
    /**
     * Ritorna il campo 'StartDate'
     *
     * @return string
     */
    public function getStartDate()
    {
        return $this->_startDate;
    }
    
    /**
     * Set value 'EndDate'
     *
     * @param string $value
     */
    public function setEndDate($value)
    {
        $this->_endDate = $value;
    }
    
    /**
     * Ritorna il campo 'EndDate'
     *
     * @return string
     */
    public function getEndDate()
    {
        return $this->_endDate;
    }
    
    /**
     * Set value 'Release'
     *
     * @param string $value
     */
    public function setRelease($value)
    {
        $this->_release = $value;
    }
    
    /**
     * Ritorna il campo 'Release'
     *
     * @return string
     */
    public function getRelease()
    {
        return $this->_release();
    }
    
    /**
     * Set value 'Options'
     *
     * @param string $value
     */
    public function setOptions($value)
    {
        $this->_options = $value;
    }
    
    /**
     * Ritorna il campo options
     *
     * @return string
     */
    public function getOptions()
    {
        return $this->_options;
    }
    
    /**
     * Set value 'SrcType'
     *
     * @param string $value
     */
    public function setSrcType($value)
    {
        $this->_srcType = $value;
    }
    
    /**
     * Ritorna il campo 'SrcType'
     *
     * @return string
     */
    public function getSrcType()
    {
        return $this->_srcType;
    }
    
    /**
     * Set value 'OpDescr'
     *
     * @param string $value
     */
    public function setOpDescr($value)
    {
        $this->_opDescr = $value;
    }
    
    /**
     * Ritorna il campo 'OpDescr'
     *
     * @return string
     */
    public function getOpDescr()
    {
        return $this->_opDescr;
    }
    
    /**
     * Set value 'Mac'
     *
     * @param string $value
     */
    public function setMac($value)
    {
        $this->_mac = $value;
    }
    
    /**
     * Ritorna il campo 'Mac'
     *
     * @return string
     */
    public function getMac()
    {
        return $this->_mac;
    }
    
    /**
     * Esegue la validazione dei campi prima dei
     * eseguire la richiesta al webservice
     *
     * @throws \Exception
     */
    protected function _validate()
    {
        if(empty($this->_operation)) {
            $this->_operation = 'LISTOPERATION';
        }
        if(empty($this->_timestamp)) {
            $this->_timestamp = $this->_date->date()->format('Y-m-d\TH:i:s.u');
            $this->_timestamp = substr($this->_timestamp, 0, -3);
        }
        if(empty($this->_reqRefNum)) {
            throw new \Exception("[ReqRefNum] param is mandatory");
        }
        if(empty($this->_startDate)) {
            throw new \Exception("[StartDate] param is mandatory");
        }
        if(empty($this->_endDate)) {
            throw new \Exception("[EndDate] param is mandatory");
        }
        if(empty($this->_shopId)) {
            throw new \Exception("[ShopId] param is mandatory");
        }
        if(empty($this->_operatorId)) {
            throw new \Exception("[OperatorId] param is mandatory");
        }
        if(empty($this->_release)) {
            $this->_release = '02';
        }
        if(empty($this->_macRequestKey)) {
            throw new \Exception("[MacRequestKey] param is mandatory");
        }
        if(empty($this->_requestUrl)) {
            throw new \Exception("[RequestUrl] param is mandatory");
        }
        if(empty($this->_hash)) {
            $this->_hash = \Cooder\Sia\Model\Config\Source\Hash::HASH_SHA1;
        }
        if(empty($this->_mac)) {
            $this->_mac = $this->_calculateMac();
        }
    }
    
    /**
     * Funzione che crea la stringa XML
     * per la richiesta
     *
     * @return string
     */
    protected function _createXml()
    {
        $xml = '';
        $xml .= '<BPWXmlRequest>';
        $xml .= '<Release>' . $this->_release . '</Release>';
        $xml .= '<Request>';
        $xml .= '<Operation>' . $this->_operation . '</Operation>';
        $xml .= '<Timestamp>' . $this->_timestamp . '</Timestamp>';
        $xml .= '<Mac>' . $this->_mac . '</Mac>';
        $xml .= '</Request>';
        $xml .= '<Data>';
        $xml .= '<ListOperation>';
        $xml .= '<Header>';
        $xml .= '<ShopID>' . $this->_shopId . '</ShopID>';
        $xml .= '<OperatorID>' . $this->_operatorId . '</OperatorID>';
        $xml .= '<ReqRefNum>' . $this->_reqRefNum . '</ReqRefNum>';
        $xml .= '</Header>';
        $xml .= '<StartDate>' . $this->_startDate . '</StartDate>';
        $xml .= '<EndDate>' . $this->_endDate . '</EndDate>';
        if(!empty($this->_srcType)) {
            $xml .= '<SrcType>' . $this->_srcType . '</SrcType>';
        }
        if(!empty($this->_opDescr)) {
            $xml .= '<OpDescr>' . $this->_opDescr . '</OpDescr>';
        }
        if(!empty($this->_options)) {
            $xml .= '<Options>' . $this->_options . '</Options>';
        }
        $xml .= '</ListOperation>';
        $xml .= '</Data>';
        $xml .= '</BPWXmlRequest>';
        
        $this->_log($xml);
        
        return $xml;
    }
    
    /**
     * Esegue il parse della risposta XML data dal webservice
     *
     * @param string $xml
     *
     * @return \Cooder\Sia\Webservice\TransactionList\Response
     */
    protected function _parseResponse($xml)
    {
        $this->_xmlParser->loadXML($xml);
        $content = $this->_xmlParser->xmlToArray();
        
        $this->_response->reset();
        
        if(array_key_exists('BPWXmlResponse', $content) == false) {
            throw new \Exception("[BPWXmlResponse] value missing in the XML response");
        }
        $content = $content['BPWXmlResponse'];
        if(array_key_exists('Result', $content) == false) {
            throw new \Exception("[Result] value missing in the XML response");
        }
        $this->_response->setResult($content['Result']);
        
        if($this->_response->getResult() == self::RESPONSE_SUCCESS_CODE) {
            if(array_key_exists('Data', $content) == false) {
                throw new \Exception("[Data] value missing in the XML response");
            }
            $content = $content['Data'];
            
            if(array_key_exists('ListOperation', $content) == false) {
                throw new \Exception("[ListOperation] value missing in the XML response");
            }
            
            if(array_key_exists('NumberOfItems', $content)) {
                $this->_response->setNumberOfItems($content['NumberOfItems']);
            }
            
            if(array_key_exists('Operation', $content) && is_array($content['Operation'])) {
                if(array_key_exists('TransactionID', $content['Operation'])) {
                    $item = $this->_getItemData($content['Operation']);
                    $this->_response->addItem($item);
                } else {
                    foreach($content['Operation'] as $itemData) {
                        $item = $this->_getItemData($itemData);
                        $this->_response->addItem($item);
                    }
                }
            }
        } elseif($this->_response->getResult() != self::RESPONSE_EMPTY_CODE) {
            throw new \Exception("[ResultError]: " . $this->_response->getErrorDesc());
        }
        
        return $this->_response;
    }
    
    /**
     * Ritorna i dati di una operazione
     *
     * @param Array<mixed> $data
     * @return \Cooder\Sia\Model\Webservice\TransactionList\ResponseItem
     */
    protected function _getItemData($data)
    {
        $item = new \Cooder\Sia\Model\Webservice\TransactionList\ResponseItem();
        if(array_key_exists('TransactionID', $data)) {
            $item->setTransactionId($data['TransactionID']);
        }
        if(array_key_exists('TimestampReq', $data)) {
            $item->setTimestampReq($data['TimestampReq']);
        }
        if(array_key_exists('TimestampElab', $data)) {
            $item->setTimestampElab($data['TimestampElab']);
        }
        if(array_key_exists('SrcType', $data)) {
            $item->setSrcType($data['SrcType']);
        }
        if(array_key_exists('Amount', $data)) {
            $item->setAmount($data['Amount']);
        }
        if(array_key_exists('Result', $data)) {
            $item->setResult($data['Result']);
        }
        if(array_key_exists('Status', $data)) {
            $item->setStatus($data['Status']);
        }
        if(array_key_exists('OpDescr', $data)) {
            $item->setOpDescr($data['OpDescr']);
        }
        if(array_key_exists('MAC', $data)) {
            $item->setMac($data['MAC']);
        }
        if(array_key_exists('Authorization', $data)) {
            $authData = $this->_getAuthData($data['Authorization']);
            $item->setAuthorization($authData);
        }
        return $item;
    }
    
    /**
     * Ritorna i dati di autorizzazione
     *
     * @param Array<mixed> $data
     * @return \Cooder\Sia\Model\Webservice\AuthorizationList\ResponseItem
     */
    protected function _getAuthData($data)
    {
        $item = new \Cooder\Sia\Model\Webservice\AuthorizationList\ResponseItem();
        if(array_key_exists('PaymentType', $data)) {
            $item->setPaymentType($data['PaymentType']);
        }
        if(array_key_exists('AuthorizationType', $data)) {
            $item->setAuthorizationType($data['AuthorizationType']);
        }
        if(array_key_exists('TransactionID', $data)) {
            $item->setTransactionId($data['TransactionID']);
        }
        if(array_key_exists('Network', $data)) {
            $item->setNetwork($data['Network']);
        }
        if(array_key_exists('OrderID', $data)) {
            $item->setOrderId($data['OrderID']);
        }
        if(array_key_exists('TransactionAmount', $data)) {
            $item->setTransactionAmount($data['TransactionAmount']);
        }
        if(array_key_exists('AuthorizedAmount', $data)) {
            $item->setAuthorizedAmount($data['AuthorizedAmount']);
        }
        if(array_key_exists('Currency', $data)) {
            $item->setCurrency($data['Currency']);
        }
        if(array_key_exists('Exponent', $data)) {
            $item->setExponent($data['Exponent']);
        }
        if(array_key_exists('AccountedAmount', $data)) {
            $item->setAccountedAmount($data['AccountedAmount']);
        }
        if(array_key_exists('TransactionResult', $data)) {
            $item->setTransactionResult($data['TransactionResult']);
        }
        if(array_key_exists('Timestamp', $data)) {
            $item->setTimestamp($data['Timestamp']);
        }
        if(array_key_exists('AuthorizationNumber', $data)) {
            $item->setAuthorizationNumber($data['AuthorizationNumber']);
        }
        if(array_key_exists('AcquirerBIN', $data)) {
            $item->setAcquirerBin($data['AcquirerBIN']);
        }
        if(array_key_exists('MerchantID', $data)) {
            $item->setMerchantId($data['MerchantID']);
        }
        if(array_key_exists('TransactionStatus', $data)) {
            $item->setTransactionStatus($data['TransactionStatus']);
        }
        if(array_key_exists('ResponseCodeISO', $data)) {
            $item->setResponseCodeISO($data['ResponseCodeISO']);
        }
        if(array_key_exists('PanTail', $data)) {
            $item->setPanTail($data['PanTail']);
        }
        if(array_key_exists('PanExpiryDate', $data)) {
            $item->setPanExpiryDate($data['PanExpiryDate']);
        }
        if(array_key_exists('PaymentTypePP', $data)) {
            $item->setPaymentTypePp($data['PaymentTypePP']);
        }
        if(array_key_exists('RRN', $data)) {
            $item->setRrn($data['RRN']);
        }
        if(array_key_exists('CardType', $data)) {
            $item->setCardType($data['CardType']);
        }
        
        return $item;
    }
    
    /**
     * Calcola il codice MAC dati i parametri
     * della richiesta
     *
     * @return string
     */
    protected function _calculateMac()
    {
        $hash = 'OPERATION='. $this->_operation . '&';
        $hash .= 'TIMESTAMP='. $this->_timestamp . '&';
        $hash .= 'SHOPID='. $this->_shopId . '&';
        $hash .= 'OPERATORID='. $this->_operatorId . '&';
        $hash .= 'REQREFNUM='. $this->_reqRefNum . '&';
        $hash .= 'STARTDATE='. $this->_startDate . '&';
        $hash .= 'ENDDATE='. $this->_endDate . '&';
        if(!empty($this->_opDescr)) {
            $hash .= 'OPDESCR='. $this->_opDescr . '&';
        }
        if(!empty($this->_options)) {
            $hash .= 'OPTIONS='. $this->_options . '&';
        }
        if($this->_hash == \Cooder\Sia\Model\Config\Source\Hash::HASH_SHA1) {
            $hash .= $this->_macRequestKey;
            return sha1($hash);
        } elseif($this->_hash == \Cooder\Sia\Model\Config\Source\Hash::HASH_MD5) {
            $hash .= $this->_macRequestKey;
            return md5($hash);
        } else {
            return hash_hmac('sha256', $hash, $this->_macRequestKey);
        }
    }
    
    /**
     * Invia la richiesta al webservice per la creazione del link di pagamento
     * e restituisce la response
     *
     * @return \Cooder\Sia\Webservice\PaymentInit\Response
     */
    public function send()
    {
        try {
            $this->_validate();
            $xml = $this->_createXml();
            
            $this->_curl->post($this->_requestUrl, ['data' => $xml]);
            $response = $this->_curl->getBody();
            
            $this->_log($response);
            
            return $this->_parseResponse($response);
        } catch(\Exception $ex) {
            throw new \Exception("ListOperation Request failed: " . $ex->getMessage());
        }
    }
    
    /**
     * Scrive log se attivato
     *
     * @param string $message
     * @param boolean $forceLog
     */
    protected function _log($message, $forceLog = false)
    {
        if ($this->_baseLogger && ($forceLog || $this->_debug)) {
            $this->_baseLogger->debug($message);
        }
    }
    
}