<?php

namespace Cooder\Sia\Cron;

/**
 * Cron per la sincronizzazione deella lista di transazioni
 * 
 * @author Devid Marcantoni <devid@cooder.it>
 */
class SyncTransactionList
{
    
    /**
	 * Oggetto per il recupero dei dati dalla configurazione
	 *
	 * @var \Magento\Framework\App\Config\ScopeConfigInterface
	 */
	protected $_scopeConfig;
    
	/**
	 * Oggetto per il recupero delle informazioni sugli store/website
	 * 
	 * @var \Magento\Store\Model\StoreManagerInterface
	 */
	protected $_storeManager;
	
    /**
     * Sync Factory
     *
     * @var \Cooder\Sia\Model\SyncFactory
     */
    protected $_syncFactory;
    
    /**
     * Transaction Factory
     * 
     * @var \Cooder\Sia\Model\TransactionFactory
     */
    protected $_transactionFactory;
    
    /**
     * Oggetto per la request al webservice
     * per il recupero della lista di transazioni
     *
     * @var \Cooder\Sia\Model\Webservice\TransactionList\Request
     */
    protected $_webserviceRequest;
    
    /**
     * Connection
     *
     * @var \Magento\Framework\DB\Adapter\AdapterInterface
     */
    protected $_connection;
    
    /**
     * Logger
     * 
     * @var \Cooder\Sia\Logger\Logger
     */
    protected $_logger;
    
    /**
     * Helper del modulo
     * 
     * @var \Cooder\Sia\Helper\Data
     */
    protected $_helper;
    
    /**
     * Date Object
     *
     * @var \Magento\Framework\Stdlib\DateTime\TimezoneInterface
     */
    protected $_date;
    
    /**
     * Order Repository
     * 
     * @var \Magento\Sales\Model\OrderFactory
     */
    protected $_orderFactory;
    
    /**
     * Credit memo Factory
     * 
     * @var \Magento\Sales\Model\Order\CreditmemoFactory
     */
    protected $_creditmemoFactory;
    
    /**
     * Credit memo Service
     * 
     * @var \Magento\Sales\Model\Service\CreditmemoService
     */
    protected $_creditmemoService;
    
   /**
    * Costruttore
    * 
    * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
    * @param \Magento\Store\Model\StoreManagerInterface $storeManager
    * @param \Cooder\Sia\Model\SyncFactory $syncFactory
    * @param \Cooder\Sia\Model\TransactionFactory $transactionFactory
    * @param \Cooder\Sia\Model\Webservice\TransactionList\Request $webserviceRequest
    * @param \Magento\Framework\App\ResourceConnection $resource
    * @param \Cooder\Sia\Logger\Logger $logger
    * @param \Cooder\Sia\Helper\Data $helper
    * @param \Magento\Framework\Stdlib\DateTime\TimezoneInterface $date
    * @param \Magento\Sales\Model\OrderFactory $orderFactory
    * @param \Magento\Sales\Model\Order\CreditmemoFactory $creditmemoFactory
    * @param \Magento\Sales\Model\Service\CreditmemoService $creditmemoService
    */
    public function __construct(
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Cooder\Sia\Model\SyncFactory $syncFactory,
        \Cooder\Sia\Model\TransactionFactory $transactionFactory,
        \Cooder\Sia\Model\Webservice\TransactionList\Request $webserviceRequest,
        \Magento\Framework\App\ResourceConnection $resource,
        \Cooder\Sia\Logger\Logger $logger,
        \Cooder\Sia\Helper\Data $helper,
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $date,
        \Magento\Sales\Model\OrderFactory $orderFactory,
        \Magento\Sales\Model\Order\CreditmemoFactory $creditmemoFactory,
        \Magento\Sales\Model\Service\CreditmemoService $creditmemoService
    )
    {
        $this->_scopeConfig = $scopeConfig;
        $this->_storeManager = $storeManager;
        $this->_syncFactory = $syncFactory;
        $this->_transactionFactory = $transactionFactory;
        $this->_webserviceRequest = $webserviceRequest;
        $this->_connection = $resource->getConnection();
        $this->_logger = $logger;
        $this->_helper = $helper;
        $this->_date = $date;
        $this->_orderFactory = $orderFactory;
        $this->_creditmemoFactory = $creditmemoFactory;
        $this->_creditmemoService = $creditmemoService;
    }
    
    /**
     * Esegue il sync delle transazioni sia
     *
     * @return boolean
     * @throws \Exception
     */
    public function execute()
    {
        $enabled = $this->_scopeConfig->getValue('payment/sia/active_transaction_sync');
        if(!$enabled) {
            $this->_logger->info("-- Transaction Sync disabled --");
            return true;
        }
        try {
            $startDate = date("Y-m-d", strtotime( '-1 days' ) );
            $endDate = date("Y-m-d", strtotime( '-1 days' ) );
            $this->_logger->info('[sync][start-date] ' . $startDate . ' [end-date] ' . $endDate);
            
            $this->_init($startDate, $endDate);
            
            $nextSync = $this->_getNextSync();
            if(!empty($nextSync)) {
                $this->_logger->info('[sync][shop-id] ' . $nextSync->getData('shop_id') . ' [start-date] ' . $nextSync->getData('start_date') . ' [end-date] ' . $nextSync->getData('end_date'));
                $this->_sync($nextSync);
            }
            
            return true;
        } catch(\Exception $ex) {
            throw new \Exception("Error during sync sia transaction list: " . $ex->getMessage());
        }
    }
    
    /**
     * Inizializzazione dei dati
     * 
     * @param string $startDate
     * @param string $endDate
     * @return boolean
     */
    protected function _init($startDate, $endDate)
    {
        $syncCollection = $this->_syncFactory->create()->getCollection();
        $syncCollection->addFieldToFilter('start_date', $startDate);
        $syncCollection->addFieldToFilter('end_date', $endDate);
        if($syncCollection->count() == 0) {
            $shopIdList = [];
            foreach($this->_storeManager->getWebsites() as $website) {
                $active = $this->_scopeConfig->getValue('payment/sia/active', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                if($active) {
                    $environment = $this->_scopeConfig->getValue('payment/sia/sandbox/environment', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                    if($environment == \Cooder\Sia\Model\Config\Source\Environment::SANDBOX) {
                        $shopId = $this->_scopeConfig->getValue('payment/sia/sandbox/shop_id_sandbox', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                    } else {
                        $shopId = $this->_scopeConfig->getValue('payment/sia/production/shop_id_production', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                    }
                    if(in_array($shopId, $shopIdList) == false) {
                        $operatorId = $this->_scopeConfig->getValue('payment/sia/operator_id', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                        $debug = $this->_scopeConfig->getValue('payment/sia/debug', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                        $hash = $this->_scopeConfig->getValue('payment/sia/hash', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                        if($environment == \Cooder\Sia\Model\Config\Source\Environment::SANDBOX) {
                            $macApiKey = $this->_scopeConfig->getValue('payment/sia/sandbox/mac_api_key_sandbox', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                            $endpoint = $this->_scopeConfig->getValue('payment/sia/sandbox/endpointxml_sandbox', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                        } else {
                            $macApiKey = $this->_scopeConfig->getValue('payment/sia/production/mac_api_key_production', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                            $endpoint = $this->_scopeConfig->getValue('payment/sia/production/endpointxml_production', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $website->getId());
                        }
                        $shopIdList[] = $shopId;
                        
                        $data = [];
                        $data['start_date'] = $startDate;
                        $data['end_date'] = $endDate;
                        $data['status'] = 0;
                        $data['shop_id'] = $shopId;
                        $data['operator_id'] = $operatorId;
                        $data['mac_api_key'] = $macApiKey;
                        $data['endpoint'] = $endpoint;
                        $data['debug'] = $debug;
                        $data['hash'] = $hash;
                        $this->_connection->insert('sia_sync_transaction', $data);
                    }
                }
            }
        }
        return true;
    }
    
    /**
     * Ritorna, se esiste, la nuova entry da sincronizzare
     * 
     * @return boolean|\Cooder\Sia\Model\Sync
     */
    protected function _getNextSync()
    {
        $syncCollection = $this->_syncFactory->create()->getCollection();
        $syncCollection->addFieldToFilter('status', 0);
        if($syncCollection->count() == 0) {
            return false;
        }
        return $syncCollection->getFirstItem();
    }
    
    /**
     * Esegue la sync in input
     * 
     * @param \Cooder\Sia\Model\Sync $sync
     */
    protected function _sync($sync)
    {
        $requestUrl = $sync->getData('endpoint');
        $shopId = $sync->getData('shop_id');
        $macApiKey = $sync->getData('mac_api_key');
        $hash = $sync->getData('hash');
        $operatorId = $sync->getData('operator_id');
        $startDate = $sync->getData('start_date');
        $endDate = $sync->getData('end_date');
        $reqRefNum = $this->_calculateReqRefNum();
        $debug = $sync->getData('debug');
        
        try {
            $this->_webserviceRequest->setRequestUrl($requestUrl);
            $this->_webserviceRequest->setMacRequestKey($macApiKey);
            $this->_webserviceRequest->setHash($hash);
            $this->_webserviceRequest->setShopId($shopId);
            $this->_webserviceRequest->setOperatorId($operatorId);
            $this->_webserviceRequest->setStartDate($startDate);
            $this->_webserviceRequest->setEndDate($endDate);
            $this->_webserviceRequest->setReqRefNum($reqRefNum);
            $this->_webserviceRequest->setDebug($debug);
            $response = $this->_webserviceRequest->send();
            
            foreach($response->getItems() as $item) {                
                /** @var $item \Cooder\Sia\Model\Webservice\TransactionList\ResponseItem */
                $data = [];
                
                try {
                    $order = $this->_orderFactory->create()->loadByIncrementId($item->getAuthorization()->getOrderId());
                    if(!$order->getId()) {
                        $this->_logger->info("Nessun ordine con id [" . $item->getAuthorization()->getOrderId() . "]. Transazione non memorizzata");
                        continue;
                    }
                } catch(\Exception $ex) {
                    $this->_logger->info("Nessun ordine con id [" . $item->getAuthorization()->getOrderId() . "]. Transazione non memorizzata");
                    continue;
                }
                
                $primaryKey = $this->_getPrimaryKey($item->getAuthorization()->getOrderId(), $shopId, $item->getAuthorization()->getTransactionId());
                if(!empty($primaryKey)) {
                    $data['id'] = $primaryKey;
                } else {
                    $data['customer'] = $order->getBillingAddress()->getFirstname() . ' ' . $order->getBillingAddress()->getLastname();
                    $data['customer_email'] = $order->getCustomerEmail();
                }
                
                $data['transaction_id'] = $item->getTransactionId();
                $data['timestamp_req'] = $item->getTimestampReq();
                $data['timestamp_elab'] = $item->getTimestampElab();
                $data['src_type'] = $item->getSrcType();
                $data['amount'] = $item->getAmount() / 100;
                $data['result'] = $item->getResult();
                $data['status'] = $item->getStatus();
                $data['auth_payment_type'] = $item->getAuthorization()->getPaymentType();
                $data['auth_transaction_id'] = $item->getAuthorization()->getTransactionId();
                $data['auth_network'] = $item->getAuthorization()->getNetwork();
                $data['auth_order_id'] = $item->getAuthorization()->getOrderId();
                $data['auth_transaction_amount'] = $item->getAuthorization()->getTransactionAmount() / 100;
                $data['auth_authorized_amount'] = $item->getAuthorization()->getAuthorizedAmount() / 100;
                $data['auth_currency'] = $this->_getCurrencyCode($item->getAuthorization()->getCurrency());
                $data['auth_exponent'] = $item->getAuthorization()->getExponent();
                $data['auth_accounted_amount'] = $item->getAuthorization()->getAccountedAmount() / 100;
                $data['auth_refunded_amount'] = $item->getAuthorization()->getRefundedAmount() / 100;
                $data['auth_transaction_result'] = $item->getAuthorization()->getTransactionResult();
                $data['auth_timestamp'] = $item->getAuthorization()->getTimestamp();
                $data['auth_authorization_number'] = $item->getAuthorization()->getAuthorizationNumber();
                $data['auth_acquirer_bin'] = $item->getAuthorization()->getAcquirerBin();
                $data['auth_merchant_id'] = $item->getAuthorization()->getMerchantId();
                $data['auth_transaction_status'] = $item->getAuthorization()->getTransactionStatus();
                $data['auth_response_code_iso'] = $item->getAuthorization()->getResponseCodeIso();
                $data['auth_pan_tail'] = $item->getAuthorization()->getPanTail();
                $data['auth_pan_tail_expiry_date'] = $item->getAuthorization()->getPanExpiryDate();
                $data['auth_payment_type_pp'] = $item->getAuthorization()->getPaymentTypePP();
                $data['auth_rrn'] = $item->getAuthorization()->getRrn();
                $data['auth_card_type'] = $item->getAuthorization()->getCardType();
                $data['shop_id'] = $shopId;
                
                $transaction = $this->_transactionFactory->create();
                $transaction->setData($data);
                $transaction->save();
                
                // Se necessario/abilitato creo la nota di credito
                $creditmemoCreationEnabled = $this->_scopeConfig->getValue('payment/sia/create_creditmemo');
                if($creditmemoCreationEnabled) {
                    $creditmemoCreationTransactionCodes = $this->_scopeConfig->getValue('payment/sia/crate_creditmemo_srctype');
                    
                    if(empty($creditmemoCreationTransactionCodes)) {
                        $creditmemoCreationTransactionCodes = [];
                    } else {
                        $creditmemoCreationTransactionCodes = explode(",", $creditmemoCreationTransactionCodes);
                    }
                             
                    if(in_array($item->getSrcType(), $creditmemoCreationTransactionCodes)) {
                        $this->_createCreditmemo($order);
                    }
                }
            } 
            
            $this->_connection->update('sia_sync_transaction', ['status' => 1], ["id = ?" => $sync->getData('id')]);
        } catch(\Exception $ex) {
            $this->_logger->info($ex->getMessage());
            $this->_connection->update('sia_sync_transaction', ['status' => 2, 'error_desc' => $ex->getMessage()], ["id = ?" => $sync->getData('id')]);
        }
    }
    
    /**
     * Calcola il valore per il campo ReqRefNum
     *
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @return string
     */
    protected function _calculateReqRefNum()
    {
        return $this->_date->date()->format('Ymd') . str_pad(time(), 24, '0', STR_PAD_LEFT);
    }
    
    /**
     * Ritorna il currency code
     * 
     * @param string $currency
     * @return string
     */
    protected function _getCurrencyCode($currency)
    {
        switch ($currency) {
            case '840': {
                return 'USD';
                break;
            }
            default: {
                return 'EUR';
                break;
            }
        }
    }
    
    /**
     * Ritorna, se esiste, identificatore numerico della entry
     * a db che rappresenta la transazione in atto.
     * 
     * @param string $orderId
     * @param string $shopId
     * @param string $authTransactionId
     * @return NULL|int
     */
    protected function _getPrimaryKey($orderId, $shopId, $authTransactionId)
    {
        $collection = $this->_transactionFactory->create()->getCollection();
        $collection->addFieldToFilter('auth_order_id', $orderId);
        $collection->addFieldToFilter('shop_id', $shopId);
        $collection->addFieldToFilter('auth_transaction_id', $authTransactionId);
        $collection->addFieldToFilter('transaction_id', ['null' => true]);
        if($collection->count() != 1) {
            return null;
        }
        return $collection->getFirstItem()->getId();
    }
    
    /**
     * Crea la nota di credito per un ordine
     * 
     * @param \Magento\Sales\Model\Order $order
     * @return \Magento\Sales\Model\Order\Creditmemo
     */
    protected function _createCreditmemo(\Magento\Sales\Model\Order $order)
    {
        $creditmemo = $this->_creditmemoFactory->createByOrder($order);
        $this->_creditmemoService->refund($creditmemo, true);
        return $creditmemo;
        
    }
}