<?php

namespace Cooder\Sia\Controller\Payment;

/**
 * Controller per la comunicazione S2S del
 * risultato di un pagamento
 * 
 * @author Devid Marcantoni <devid@cooder.it>
 */
class Notify extends \Magento\Framework\App\Action\Action
{
    
    /**
     * Factory per il recupero dei dati sugli ordini
     *
     * @var \Magento\Sales\Model\OrderFactory
     */
    protected $_orderFactory;
    
    /**
     * Logger del modulo
     *
     * @var \Cooder\Sia\Logger\Logger
     */
    protected $_baseLogger;
    
    /**
     * Scope Config
     *
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $_scopeConfig;
    
    /**
     * Transaction Factory
     *
     * @var \Cooder\Sia\Model\TransactionFactory
     */
    protected $_transactionFactory;
    
    /**
     * Costruttore
     * 
     * @param \Magento\Framework\App\Action\Context $context
     * @param \Magento\Sales\Model\OrderFactory $orderFactory
     * @param \Cooder\Sia\Logger\Logger $baseLogger
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Cooder\Sia\Model\TransactionFactory $transactionFactory
     */
    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        \Magento\Sales\Model\OrderFactory $orderFactory,
        \Cooder\Sia\Logger\Logger $baseLogger,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Cooder\Sia\Model\TransactionFactory $transactionFactory
    )
    {
        parent::__construct($context);
        $this->_orderFactory = $orderFactory;
        $this->_baseLogger = $baseLogger;
        $this->_scopeConfig = $scopeConfig;
        $this->_transactionFactory = $transactionFactory;
    }
    
    /**
     * Esecuzione delle verifiche sulla risposta di
     * pagamento dal server Sia S2S
     */
    public function execute()
    {
        $this->_baseLogger->debug('S2S Verify result: start');
        
        $orderIncrementId = $this->getRequest()->getParam('ORDERID');
        if(empty($orderIncrementId)) {
            $this->_baseLogger->debug('S2S Verify result: [ORDERID] missing');
            return false;
        }
        
        $order = $this->_orderFactory->create()->loadByIncrementId($orderIncrementId);
        if(!($order->getId())) {
            $this->_baseLogger->debug('S2S Verify result: [ORDERID] invalid');
            return false;
        }
        
        $payment = $order->getPayment();
        if (!$payment instanceof \Magento\Payment\Model\InfoInterface) {
            $this->_baseLogger->debug('Payment object is not a PaymentInfo!');
            return false;
        }
        
        $paymentMethodInstance = $payment->getMethodInstance();
        if (!$paymentMethodInstance instanceof \Cooder\Sia\Model\SiaPay) {
            $this->_baseLogger->debug('Notification is not about an SiaPay payment! Payment method object: ' . get_class($paymentMethodInstance));
            return false;
        }
        
        if($order->getState() != \Magento\Sales\Model\Order::STATE_PENDING_PAYMENT) {
            $this->_baseLogger->debug('Order Payment already processed');
            return false;
        }
        
        $result = $paymentMethodInstance->paymentVerify($this->getRequest()->getParams(), true);
        if ($result) {
            $this->_baseLogger->debug('S2S Verify result: success');
        } else {
            $this->_baseLogger->debug('S2S Verify result: failure');
        }
        
        // Salvo le informazioni sulla transazione
        $this->_saveTransaction($this->getRequest()->getParams(), $order);
        
        return true;
    }
    
    /**
     * Salva le informazioni sulla transazione
     *
     * @param Array<string> $params
     * @param \Magento\Sales\Model\Order $order
     */
    protected function _saveTransaction($params, $order)
    {
        if(array_key_exists('ORDERID', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [ORDERID]"));
        }
        if(array_key_exists('SHOPID', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [SHOPID]"));
        }
        if(array_key_exists('AUTHNUMBER', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [AUTHNUMBER]"));
        }
        if(array_key_exists('AMOUNT', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [AMOUNT]"));
        }
        if(array_key_exists('CURRENCY', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [CURRENCY]"));
        }
        if(array_key_exists('TRANSACTIONID', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [TRANSACTIONID]"));
        }
        if(array_key_exists('RESULT', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [RESULT]"));
        }
        if(array_key_exists('NETWORK', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [NETWORK]"));
        }
        
        $primaryKey = $this->_getPrimaryKey($params['ORDERID'], $params['SHOPID']);
        if(empty($primaryKey)) {
            $transaction = $this->_transactionFactory->create();
            $transaction->setData('auth_order_id', $params['ORDERID']);
            $transaction->setData('customer', $order->getBillingAddress()->getFirstname() . ' ' . $order->getBillingAddress()->getLastname());
            $transaction->setData('customer_email', $order->getCustomerEmail());
            $transaction->setData('shop_id', $params['SHOPID']);
            $transaction->setData('auth_transaction_id', $params['TRANSACTIONID']);
            $transaction->setData('auth_authorization_number', $params['AUTHNUMBER']);
            $transaction->setData('auth_network', $params['NETWORK']);
            $transaction->setData('auth_transaction_amount', $params['AMOUNT'] / 100);
            $transaction->setData('auth_authorized_amount', $params['AMOUNT'] / 100);
            $transaction->setData('amount', $params['AMOUNT'] / 100);
            $transaction->setData('auth_currency', $this->_getCurrencyCode($params['CURRENCY']));
            $transaction->setData('auth_transaction_result', $params['RESULT']);
            
            if(array_key_exists('ACCOUNTINGMODE', $params) == false) {
                $transaction->setData('auth_accounting_mode', 'I');
            } else {
                $transaction->setData('auth_accounting_mode', $params['ACCOUNTINGMODE']);
            }
            if(array_key_exists('AUTHORMOD', $params) == false) {
                $transaction->setData('auth_author_mode', 'I');
            } else {
                $transaction->setData('auth_author_mode', $params['AUTHORMOD']);
            }
            
            $transaction->save();
        }
    }
    
    /**
     * Ritorna il currency code
     *
     * @param string $currency
     * @return string
     */
    protected function _getCurrencyCode($currency)
    {
        switch ($currency) {
            case '840': {
                return 'USD';
                break;
            }
            default: {
                return 'EUR';
                break;
            }
        }
    }
    
    /**
     * Ritorna, se esiste, identificatore numerico della entry
     * a db che rappresenta la transazione in atto.
     *
     * @param string $orderId
     * @param string $shopId
     * @return NULL|int
     */
    protected function _getPrimaryKey($orderId, $shopId)
    {
        $collection = $this->_transactionFactory->create()->getCollection();
        $collection->addFieldToFilter('auth_order_id', $orderId);
        $collection->addFieldToFilter('shop_id', $shopId);
        if($collection->count() == 0) {
            return null;
        }
        return $collection->getFirstItem()->getId();
    }
}