<?php

namespace Cooder\Sia\Controller\Payment;

/**
 * Controller per la pagina di risultato pagamento mediante SiaPay
 * 
 * @author Devid Marcantoni <devid@cooder.it>
 */
class Done extends \Magento\Framework\App\Action\Action
{
    
    /**
     * Checkout Session
     * 
     * @var \Magento\Checkout\Model\Session
     */
    protected $_checkoutSession;
    
    /**
     * Logger del modulo
     * 
     * @var \Cooder\Sia\Logger\Logger
     */
    protected $_baseLogger;
    
    /**
     * Scope Config
     *
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $_scopeConfig;
    
    /**
     * Transaction Factory
     *
     * @var \Cooder\Sia\Model\TransactionFactory
     */
    protected $_transactionFactory;
    
    /**
     * Store Manager
     *
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $_storeManager;
    
    /**
     * Costruttore
     * 
     * @param \Magento\Framework\App\Action\Context $context
     * @param \Magento\Checkout\Model\Session $checkoutSession
     * @param \Cooder\Sia\Logger\Logger $baseLogger
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Cooder\Sia\Model\TransactionFactory $transactionFactory
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     */
    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Cooder\Sia\Logger\Logger $baseLogger,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Cooder\Sia\Model\TransactionFactory $transactionFactory,
        \Magento\Store\Model\StoreManagerInterface $storeManager
    ) 
    {
        parent::__construct($context);
        $this->_checkoutSession = $checkoutSession;
        $this->_baseLogger = $baseLogger;
        $this->_scopeConfig = $scopeConfig;
        $this->_transactionFactory = $transactionFactory;
        $this->_storeManager = $storeManager;
    }
    
    /**
     * Esecuzione delle verifiche sulla risposta di 
     * pagamento dal server Sia
     */
    public function execute()
    {
        $this->_baseLogger->debug('Verify result: start');
        
        // Recupero ordine in sessione
        $lastOrder = $this->_checkoutSession->getLastRealOrder();
        
        // Istanzio metodo di pagamento
        $payment = $lastOrder->getPayment();
        if (!$payment instanceof \Magento\Payment\Model\InfoInterface) {
            $this->_baseLogger->debug('Payment object is not a PaymentInfo!');
            return $this->_sendErrorResponse();
        }
        
        $paymentMethodInstance = $payment->getMethodInstance();
        if (!$paymentMethodInstance instanceof \Cooder\Sia\Model\SiaPay) {
            $this->_baseLogger->debug(
                'Notification is not about an SiaPay payment! Payment method object: '.
                get_class($paymentMethodInstance)
            );
            return $this->_sendErrorResponse();
        }
        
        // Recupero parametri in sessione
        $params = $this->getRequest()->getParams();
                
        // Eseguo verifica dei dati di pagamento
        $result = $paymentMethodInstance->paymentVerify($params);
        
        // Salvo le informazioni sulla transazione
        $this->_saveTransaction($params, $lastOrder);
        
        // Eseguo redirect a seconda del risultato
        if ($result) {
            $this->_baseLogger->debug('Verify result: success');
            $url = $this->_scopeConfig->getValue('payment/sia/url_success', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $this->_storeManager->getWebsite()->getId());
            if(empty($url)) {
                return $this->_redirect('checkout/onepage/success');
            } else {
                return $this->_redirect($url);
            }
        }
        
        $this->_baseLogger->debug('Verify result: error');
        $url = $this->_scopeConfig->getValue('payment/sia/url_failure', \Magento\Store\Model\ScopeInterface::SCOPE_WEBSITE, $this->_storeManager->getWebsite()->getId());
        if(empty($url)) {
            return $this->_redirect('checkout/onepage/failure');
        } else {
            return $this->_redirect($url);
        }
    }
    
    /**
     * Invia response di errore
     * 
     * @return \Magento\Framework\App\ResponseInterface
     */
    protected function _sendErrorResponse()
    {
        $response = $this->getResponse();
        $response->setHttpResponseCode(500);
        return $response;
    }
    
    /**
     * Salva le informazioni sulla transazione
     * 
     * @param Array<string> $params
     * @param \Magento\Sales\Model\Order $order
     */
    protected function _saveTransaction($params, $order)
    {
        if(array_key_exists('ORDERID', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [ORDERID]"));
        }
        if(array_key_exists('SHOPID', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [SHOPID]"));
        }
        if(array_key_exists('AUTHNUMBER', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [AUTHNUMBER]"));
        }
        if(array_key_exists('AMOUNT', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [AMOUNT]"));
        }
        if(array_key_exists('CURRENCY', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [CURRENCY]"));
        }
        if(array_key_exists('TRANSACTIONID', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [TRANSACTIONID]"));
        }
        if(array_key_exists('RESULT', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [RESULT]"));
        }
        if(array_key_exists('NETWORK', $params) == false) {
            $this->_baseLogger->info(__("Impossibile to save transaction. Missing param [NETWORK]"));
        }
        
        $primaryKey = $this->_getPrimaryKey($params['ORDERID'], $params['SHOPID']);
        if(empty($primaryKey)) {
            $transaction = $this->_transactionFactory->create();
            $transaction->setData('auth_order_id', $params['ORDERID']);
            $transaction->setData('customer', $order->getBillingAddress()->getFirstname() . ' ' . $order->getBillingAddress()->getLastname());
            $transaction->setData('customer_email', $order->getCustomerEmail());
            $transaction->setData('shop_id', $params['SHOPID']);
            $transaction->setData('auth_transaction_id', $params['TRANSACTIONID']);
            $transaction->setData('auth_authorization_number', $params['AUTHNUMBER']);
            $transaction->setData('auth_network', $params['NETWORK']);
            $transaction->setData('auth_transaction_amount', $params['AMOUNT'] / 100);
            $transaction->setData('auth_authorized_amount', $params['AMOUNT'] / 100);
            $transaction->setData('amount', $params['AMOUNT'] / 100);
            $transaction->setData('auth_currency', $this->_getCurrencyCode($params['CURRENCY']));
            $transaction->setData('auth_transaction_result', $params['RESULT']);
            
            if(array_key_exists('ACCOUNTINGMODE', $params) == false) {
                $transaction->setData('auth_accounting_mode', 'I');
            } else {
                $transaction->setData('auth_accounting_mode', $params['ACCOUNTINGMODE']);
            }
            
            if(array_key_exists('AUTHORMOD', $params) == false) {
                $transaction->setData('auth_author_mode', 'I');
            } else {
                $transaction->setData('auth_author_mode', $params['AUTHORMOD']);
            }
            
            $transaction->save();
        }
    }
    
    /**
     * Ritorna il currency code
     *
     * @param string $currency
     * @return string
     */
    protected function _getCurrencyCode($currency)
    {
        switch ($currency) {
            case '840': {
                return 'USD';
                break;
            }
            default: {
                return 'EUR';
                break;
            }
        }
    }
    
    /**
     * Ritorna, se esiste, identificatore numerico della entry
     * a db che rappresenta la transazione in atto.
     *
     * @param string $orderId
     * @param string $shopId
     * @return NULL|int
     */
    protected function _getPrimaryKey($orderId, $shopId)
    {
        $collection = $this->_transactionFactory->create()->getCollection();
        $collection->addFieldToFilter('auth_order_id', $orderId);
        $collection->addFieldToFilter('shop_id', $shopId);
        if($collection->count() == 0) {
            return null;
        }
        return $collection->getFirstItem()->getId();
    }
}