<?php

namespace Cooder\Sia\Block;

/**
 * BLocco per il redirect alla pagina di pagamento Sia
 * 
 * @author Devid Marcantoni <devid@cooder.it>
 */
class RedirectForm extends \Magento\Framework\View\Element\Template
{
    
    const CURRENCY_CODE_EUR = '978';
    const CURRENCY_CODE_USD = '840';
    const CURRENCY_EXPONENT_USD = '2';
    
   /**
     * Checkout Session
     *
     * @var \Magento\Checkout\Model\Session
     */
    protected $_checkoutSession;
    
    /**
     * Order Factory
     *
     * @var \Magento\Sales\Model\OrderFactory
     */
    protected $_orderFactory;
    
    /**
     * Scope Config
     * 
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $_scopeConfig;
    
    /**
     * Classe per le conversioni di valute
     * 
     * @var \Magento\Directory\Helper\Data
     */
    protected $_directoryHelper;
    
    /**
     * Price Currency
     *
     * @var \Magento\Framework\Pricing\PriceCurrencyInterface
     */
    protected $_priceCurrency;
    
    /**
     * Store Manager
     *
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $_storeManager;
    
    /**
     * Builder per Url
     * 
     * @var \Magento\Framework\UrlInterface
     */
    protected $_urlBuilder;
    
    /**
     * Helper Data
     * 
     * @var \Cooder\Sia\Helper\Data
     */
    protected $_helper;
    
    /**
     * Ordine corrente
     * 
     * @var \Magento\Sales\Model\Order
     */
    protected $_order;
        
   /**
    * Costruttore
    * 
    * @param \Magento\Framework\View\Element\Template\Context $context
    * @param \Magento\Checkout\Model\Session $checkoutSession
    * @param \Magento\Sales\Model\OrderFactory $orderFactory
    * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
    * @param \Magento\Directory\Helper\Data $directoryHelper
    * @param \Magento\Framework\Pricing\PriceCurrencyInterface $priceCurrency
    * @param \Magento\Store\Model\StoreManagerInterface $storeManager
    * @param \Magento\Framework\UrlInterface $urlBuilder
    * @param \Cooder\Sia\Helper\Data $helper
    */
	public function __construct(
	    \Magento\Framework\View\Element\Template\Context $context,
	    \Magento\Checkout\Model\Session $checkoutSession,
	    \Magento\Sales\Model\OrderFactory $orderFactory,
	    \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
	    \Magento\Directory\Helper\Data $directoryHelper,
	    \Magento\Framework\Pricing\PriceCurrencyInterface $priceCurrency,
	    \Magento\Store\Model\StoreManagerInterface $storeManager,
	    \Magento\Framework\UrlInterface $urlBuilder,
	    \Cooder\Sia\Helper\Data $helper
	)
	{
		parent::__construct($context);
		$this->_checkoutSession = $checkoutSession;
		$this->_orderFactory = $orderFactory;
		$this->_scopeConfig = $scopeConfig;
		$this->_directoryHelper = $directoryHelper;
		$this->_priceCurrency = $priceCurrency;
		$this->_storeManager = $storeManager;
		$this->_urlBuilder = $urlBuilder;
		$this->_helper = $helper;
	}

	/**
	 * Restituisce ordine corrente
	 * 
	 * @return \Magento\Sales\Model\Order
	 */
	public function getOrder()
	{
		if(empty($this->_order)) {
		    $this->_order = $this->_orderFactory->create()->loadByIncrementId($this->_checkoutSession->getLastRealOrderId());
		}
		return $this->_order;
	}
	
	/**
	 * Ritorna la url della pagina di pagamento SiaPay
	 * 
	 * @return string
	 */
	public function getFormUrl()
	{
	    return $this->_helper->getEndpoint();
	}
	
	/**
	 * Ritorna importo da pagare
	 * 
	 * @return double
	 */
	public function getAmount()
	{
	    return $this->_priceCurrency->round(
	        $this->_directoryHelper->currencyConvert(
	            $this->getOrder()->getBaseGrandTotal(),
	            $this->getOrder()->getBaseCurrencyCode(),
	            $this->_currencyCodeConversion($this->_scopeConfig->getValue('payment/sia/currency', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->_storeManager->getStore()->getId()))
	        )
	    ) * 100;
	}
	
	/**
	 * Ritorna la valuta del metodo di pagamento
	 * 
	 * @return string
	 */
	public function getCurrency()
	{
	    return $this->_scopeConfig->getValue('payment/sia/currency', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->_storeManager->getStore()->getId());
	}
	
	/**
	 * Ritorna eventuale esponente della valuta
	 * 
	 * @return string
	 */
	public function getExponent()
	{
	    $currency = $this->getCurrency();
	    if($currency == self::CURRENCY_CODE_USD) {
	        return self::CURRENCY_EXPONENT_USD;
	    }
	    return '';
	}
	
	/**
	 * Ritorna identificatore negozio
	 * 
	 * @return string
	 */
	public function getShopId()
	{
	   return $this->_helper->getShopId();
	}
	
	/**
	 * Ritorna il Mac Redirect Key
	 * 
	 * @return string
	 */
	public function getMacRedirectKey()
	{
	    return $this->_helper->getMacRedirectKey();
	}
	
	/**
	 * Ritorna identificatore ordine
	 * 
	 * @return string
	 */
	public function getOrderId()
	{
	    return $this->getOrder()->getIncrementId();
	}
	
	/**
	 * Ritorna url per pagamento annullato
	 * 
	 * @return string
	 */
	public function getUrlBack()
	{
	    return $this->_storeManager->getStore()->getBaseUrl() . 'sia/payment/cancel';
	}
	
	/**
	 * Ritorna url per pagamento andato a buon fine
	 *
	 * @return string
	 */
	public function getUrlDone()
	{
	    return $this->_storeManager->getStore()->getBaseUrl() . 'sia/payment/done';
	}
	
	/**
	 * Ritorna url ms
	 *
	 * @return string
	 */
	public function getUrlMs()
	{
	    return $this->_storeManager->getStore()->getBaseUrl() . 'sia/payment/notify';
	}
	
	/**
	 * Ritorna se pagamento immediato o differito (tpay)
	 * 
	 * @return string
	 */
	public function getAccountingMode()
	{
	    $value = $this->_scopeConfig->getValue('payment/sia/payment_action_group/payment_action', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->_storeManager->getStore()->getId());
	    if($value == \Magento\Payment\Model\Method\AbstractMethod::ACTION_AUTHORIZE) {
	        return 'D';
	    } else {
	        return 'I';
	    }
	}
	
	/**
	 * Ritorna se pagamento immediato o differito (mpay)
	 *
	 * @return string
	 */
	public function getAuthorMode()
	{
	    $value = $this->_scopeConfig->getValue('payment/sia/payment_action_author_mode', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->_storeManager->getStore()->getId());
	    if($value == \Magento\Payment\Model\Method\AbstractMethod::ACTION_AUTHORIZE) {
	        return 'D';
	    } else {
	        return 'I';
	    }
	}
	
	/**
	 * Ritorna la lingua da utilizzare nella form
	 * 
	 * @return string
	 */
	public function getLang()
	{
	    return $this->_scopeConfig->getValue('payment/sia/language', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->_storeManager->getStore()->getId());
	}
	
	/**
	 * Ritorna indirizzo email dello shop
	 * 
	 * @return string
	 */
	public function getShopEmail()
	{
	    return $this->_scopeConfig->getValue('payment/sia/shop_email', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->_storeManager->getStore()->getId());
	}
	
	/**
	 * Ritorna eventuali giorni di distanza di scadenza 
	 * della carta per la quale non accettare il pagamento
	 * 
	 * @return string
	 */
	public function getRemainingDuration()
	{
	    return $this->_scopeConfig->getValue('payment/sia/remaining_duration', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->_storeManager->getStore()->getId());
	}
	
	/**
	 * Ritorna email del customer
	 * 
	 * @return string
	 */
	public function getEmail()
	{
	    return $this->getOrder()->getCustomerEmail();
	}
	
	/**
	 * Ritorna le opzioni per la form
	 * 
	 * @return string
	 */
	public function getOptions()
	{
	    $options = 'G';
	    $remainingDuration = $this->getRemainingDuration();
	    if(!empty($remainingDuration)) {
	        $options .= 'D';
	    }
	    return $options;
	}
	
	/**
	 * Calcola il mac per la validazione della richiesta
	 * 
	 * @return string
	 */
	public function getMac()
	{
	    $hash = 'URLMS='. $this->getUrlMs() . '&';
	    $hash .= 'URLDONE='. $this->getUrlDone() . '&';
	    $hash .= 'ORDERID='. $this->getOrderId() . '&';
	    $hash .= 'SHOPID='. $this->getShopId() . '&';
	    $hash .= 'AMOUNT='. $this->getAmount() . '&';
	    $hash .= 'CURRENCY='. $this->getCurrency() . '&';
	    $exponent = $this->getExponent();
	    if(!empty($exponent)) {
	        $hash .= 'EXPONENT='. $exponent . '&';
	    }
	    $hash .= 'ACCOUNTINGMODE='. $this->getAccountingMode() . '&';
	    $hash .= 'AUTHORMODE='. $this->getAuthorMode() . '&';
	    $hash .= 'OPTIONS='. $this->getOptions() . '&';
	    $remainingDuration = $this->getRemainingDuration();
	    if(!empty($remainingDuration)) {
	        $hash .= 'REMAININGDURATION='. $remainingDuration . '&';
	    }
	    
	    $hashMode = $this->_scopeConfig->getValue('payment/sia/hash', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->_storeManager->getStore()->getId());
	    $macRedirectKey = $this->getMacRedirectKey();
	    
	    if($hashMode == \Cooder\Sia\Model\Config\Source\Hash::HASH_SHA1) {
	        $hash .= $macRedirectKey;
	        return sha1($hash);
	    } elseif($hashMode == \Cooder\Sia\Model\Config\Source\Hash::HASH_MD5) {
	        $hash .= $macRedirectKey;
	        return md5($hash);
	    } else {
	        return hash_hmac('sha256', $hash, $macRedirectKey);
	    }
	}
	
	/**
	 * Conversioni tra codici di valuta
	 *
	 * @param string $currencyCode
	 * @return string
	 */
	protected function _currencyCodeConversion($currencyCode)
	{
	    switch($currencyCode) {
	        case self::CURRENCY_CODE_EUR: {
	            return 'EUR';
	            break;
	        }
	        case self::CURRENCY_CODE_USD: {
	            return 'USD';
	            break;
	        }
	    }
	}
	
}