<?php

namespace Cooder\Sia\Model\Webservice\PaymentInit;

/**
 * Oggetto per gestione della richiesta del link per il pagamento
 * 
 * @author Devid Marcantoni <devid@cooder.it>
 */
class Request
{
    
    const RESPONSE_SUCCESS_CODE = '00';
    
    /**
     * Date Object
     * 
     * @var \Magento\Framework\Stdlib\DateTime\TimezoneInterface
     */
    protected $_date;
    
    /**
     * Oggetto per le comunicazioni cURL
     * 
     * @var \Magento\Framework\HTTP\Client\Curl
     */
    protected $_curl;
    
    /**
     * Xml Parser
     * 
     * @var \Magento\Framework\Xml\Parser
     */
    protected $_xmlParser;
    
    /**
     * Oggetto per la memorizzazione della response
     * 
     * @var \Cooder\Sia\Model\Webservice\PaymentInit\Response
     */
    protected $_response;
    
    /**
     * Logger del modulo
     * 
     * @var \Cooder\Sia\Logger\Logger
     */
    protected $_baseLogger;
    
    /**
     * Definisce se si è in modalita debug
     * 
     * @var boolean
     */
    protected $_debug = false;
 
    /**
     * Url del webservice per la creazione del link
     * per il pagamento
     * 
     * @var string
     */
    protected $_requestUrl;
    
    /**
     * Chiave per la costruzione del 
     * codice MAC
     * 
     * @var string
     */
    protected $_macRequestKey;
    
    /**
     * Metodologia di criptazione da utilizzare 
     * per la creazione del codice MAC
     * 
     * @var string
     */
    protected $_hash = \Cooder\Sia\Model\Config\Source\Hash::HASH_SHA1;
    
    /**
     * Transaction requested: to be filled in with "CREATELINK"
     * 
     * @var string
     */
    protected $_operation = 'CREATELINK';
    
    /**
     * Local timestamp of the yyyy-MM-ddTHH:mm:ss.SSS type
     * 
     * @var string
     */
    protected $_timestamp;
    
    /**
     * Unique identifier of the request managed by the merchant. 
     * It can be used to retrieve information on the request made also in the case of no  response.  
     * The  first  8  digits  must  be  in  the  yyyyMMdd  format including the request date. 
     * 
     * @var string
     */
    protected $_reqRefNum;
    
    /**
     * Identifier of the merchant’s store assigned by SIA [MID]
     * 
     * @var string
     */
    protected $_shopId;
    
    /**
     * It indicates the person who has requested the transaction
     *
     * @var string
     */
    protected $_operatorId;
    
    /**
     * It can have the following values: S (yes) or N (no). If S, 
     * the system will send the email and create the link at the same time
     *
     * @var boolean
     */
    protected $_sendMail = false;
    
    /**
     * Release of APIs: to be filled in with “02”
     *
     * @var string
     */
    protected $_release = '02';
    
    /**
     * The field OPTIONS permits to activate various additional options for the ongoing message.
     *
     * @var string
     */
    protected $_options;
    
    /**
     * Request  signature  field:  it  makes  the  data  sent unchangeable  by third parties
     *
     * @var string
     */
    protected $_mac;
    
    /**
     * Indication of the date of the end of validity of the link in the format yyyy-MM-ddTHH:mm:ss.SSS,  
     * if  it  has  no  value,  the  validity indicated in the store registered data will be used
     *
     * @var string
     */
    protected $_linkExpirationDate;
    
    /**
     * Amount expressed in the minimum unit of the value (Euro cents). Minimum length 2 maximum length 8
     *
     * @var string
     */
    protected $_linkAmount;
    
    /**
     * Currency: ISO code (EUR = 978)
     *
     * @var string
     */
    protected $_linkCurrency;
    
    /**
     * Exponent of the chosen Currency
     *
     * @var string
     */
    protected $_linkExponent;
    
    /**
     * Single identifier of the order.Its uniqueness must be guaranteed for at least 5 years. Admitted characters include letters, figures, 
     *  "-"  and  "_".  The  regular  expression  [a-zA-Z0-9\-_]  is  applied. 
     * It  is not  possible  to  request  that  a  link  be  created  for an  order  number already used for a link than can still be used
     *
     * @var string
     */
    protected $_linkOrderId;
    
    /**
     * Complete  URL  to  which  the  client’s  browser  is  to  be  redirected upon  successful  outcome  of  
     * the  transaction  (it  may include  all  of the  parameters  to  be  passed).  
     * The  system  will  hang the  outcome parameters  on  it.  Maximum  length  254  characters.  
     * In  case  of absence  of  the  parameter  the  outcome  page  will  not show  the related button. 
     *
     * @var string
     */
    protected $_linkUrlDone;
    
    /**
     * URL of the merchant system toward which SIA carries out the GET or  POST  confirming  the  effected  payment  
     * (it  may  contain  any parameters  set  by  the  store). 
     * The  system  hangs  the outcome parameters  on  it.  In  case  of  absence  of  the  parameter  the  system will  
     * not  send  the  payment  outcome  to  the  operator  via  server-to-server communication
     *
     * @var string
     */
    protected $_linkUrlMs;
    
    /**
     * Type of accounting to be used for this order (D/I)
     *
     * @var string
     */
    protected $_linkAccountingMode;
    
    /**
     * Immediate authorization only is available. The field must have the value of I. 
     *
     * @var string
     */
    protected $_linkAuthorMode = 'I';
    
    /**
     * Language in which the messages of interaction with the final user must be shown. 
     * The field is optional; the default language is Italian.The following are currently available:  
     * ITA Italian   EN English
     *
     * @var string
     */
    protected $_linkLang;
    
    /**
     * It contains the email address to which the transaction outcome is to be  sent.  
     * If  it  is  not  present,  the  one  available  in  the  store  data registered at SIA will be used. 
     * Minimum length 7 alphanumerical characters maximum length 50
     *
     * @var string
     */
    protected $_linkShopEmail;
    
    /**
     * It  contains  the  indicators  of  the  additional  options  that  are  to  be activated  for  the  payment  under  way.  
     * The  order  in  which  the options appear is irrelevant. The content of the field is not case  sensitive.
     * The value of W cannot be used via PBL. 
     * If options G and N are requested, the URLDONE field will become compulsory. The system assumes that option B is always present
     *
     * @var string
     */
    protected $_linkOptions;
    
    /**
     * Amount of commission on the service expressed in the  miminum unit of the value (EURO cents). 
     * Bear in mind that the AMOUNT parameter includes the commission.  
     * NOTE: the COMMIS parameter, if present, is relevant if and only if the additional option OPTIONS = F has also been set (payment with commission on service).
     *  Minimum length 1 maximum length 8.
     *
     * @var string
     */
    protected $_linkCommis;
    
    /**
     * Email address of client addressee of the link. If the field is not present, it will be requested from the user together with the credit card data.  
     * Compulsory if the field SENDMAIL is "S". Minimum length 7 alphanumerical characters maximum length 50.
     *
     * @var string
     */
    protected $_linkEmail;
    
    /**
     * Name of holder of payment instrument
     *
     * @var string
     */
    protected $_linkName;
    
    /**
     * Last name of holder of payment instrument 
     *
     * @var string
     */
    protected $_linkSurname;
    
    /**
     * Description of order (see OPTIONS O and V).
     * Maximum length 140
     *
     * @var string
     */
    protected $_linkOrdDescr;
    
    /**
     * Additional  description  of  accounting  transaction,  at  operator’s discretion (for immediate accounting only).
     * Maximum length 100
     *
     * @var string
     */
    protected $_linkOpDescr;
    
    /**
     * Telephone  number  to  be  assigned  beforehand  for  transactions carried out with the Jiffy circuit  
     *
     * @var string
     */
    protected $_linkPhoneNumber;
    
    /**
     * Minimum number of months of residual duration of the card used (associated with option D). Number from 0 to 60. 
     *
     * @var string
     */
    protected $_linkRemainingDuration;
    
    /**
     * User identifier generated by merchant associated with payment  
     *
     * @var string
     */
    protected $_linkUserId;
    
    /**
     * Costruttore
     * 
     * @param \Magento\Framework\Stdlib\DateTime\TimezoneInterface $date
     * @param \Magento\Framework\HTTP\Client\Curl $curl
     * @param \Magento\Framework\Xml\Parser $xmlParser
     * @param \Cooder\Sia\Model\Webservice\PaymentInit\Response $response
     * @param \Cooder\Sia\Logger\Logger $baseLogger
     */
    public function __construct(
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $date,
        \Magento\Framework\HTTP\Client\Curl $curl,
        \Magento\Framework\Xml\Parser $xmlParser,
        \Cooder\Sia\Model\Webservice\PaymentInit\Response $response,
        \Cooder\Sia\Logger\Logger $baseLogger
    )
    {
        $this->_date =  $date;
        $this->_curl = $curl;
        $this->_xmlParser = $xmlParser;
        $this->_response = $response;
        $this->_baseLogger = $baseLogger;
    }
        
    /**
     * Imposta se si è in modalita debug
     * 
     * @param boolean $value
     */
    public function setDebug($value)
    {
        $this->_debug = $value;
    }
    
    /**
     * Imposta la url del webservice per la richiesta
     * del link di pagamento
     * 
     * @param string $value
     */
    public function setRequestUrl($value)
    {
        $this->_requestUrl = $value;
    }
    
    /**
     * Ritorna il campo 'RequestUrl'
     * 
     * @return string
     */
    public function getRequestUrl()
    {
        return $this->_requestUrl;
    }
    
    
    /**
     * Imposta la chiave per la creazione del codice 
     * MAC
     *
     * @param string $value
     */
    public function setMacRequestKey($value)
    {
        $this->_macRequestKey = $value;
    }
    
    /**
     * Ritorna il campo 'MacRequestKey'
     *
     * @return string
     */
    public function getMacRequestKey()
    {
        return $this->_macRequestKey;
    }
    
    /**
     * Imposta la tecnologia di criptazione
     * per la generazione del codice MAC
     * 
     * @param string $value
     */
    public function setHash($value)
    {
        $this->_hash = $value;
    }
    
    /**
     * Ritorna il campo 'Hash'
     *
     * @return string
     */
    public function getHash()
    {
        return $this->_hash;
    }
    
    /**
     * Set value 'Operation'
     * 
     * @param string $value
     */
    public function setOperation($value)
    {
        $this->_operation = $value;
    }
    
    /**
     * Ritorna il campo 'Operation'
     * 
     * @return string
     */
    public function getOperation()
    {
        return $this->_operation;
    }
    
    /**
     * Set value 'Timestamp'
     *
     * @param string $value
     */
    public function setTimestamp($value)
    {
        $this->_timestamp = $value;
    }
    
    /**
     * Ritorna il campo 'Timestamp'
     * 
     * @return string
     */
    public function getTimestamp()
    {
        return $this->_timestamp;
    }
    
    /**
     * Set value 'ReqRefNum'
     *
     * @param string $value
     */
    public function setReqRefNum($value)
    {
        $this->_reqRefNum = $value;
    }
    
    /**
     * Ritorna il campo 'ReqRefNum'
     * 
     * @return string
     */
    public function getReqRefNum()
    {
        return $this->_reqRefNum;
    }
    
    /**
     * Set value 'ShopId'
     *
     * @param string $value
     */
    public function setShopId($value)
    {
        $this->_shopId = $value;
    }
    
    /**
     * Ritorna il campo 'ShopId'
     * 
     * @return string
     */
    public function getShopId()
    {
        return $this->_shopId;
    }
    
    /**
     * Set value 'OperatorId'
     *
     * @param string $value
     */
    public function setOperatorId($value)
    {
        $this->_operatorId = $value;
    }
    
    /**
     * Ritorna il campo 'OperatorId'
     * 
     * @return string
     */
    public function getOperatorId()
    {
        return $this->_operatorId;
    }
    
    /**
     * Set value 'SendMail'
     *
     * @param string $value
     */
    public function setSendMail($value)
    {
        $this->_sendMail = $value;
    }
    
    /**
     * Ritorna il campo 'SendMail'
     * 
     * @return boolean
     */
    public function getSendMail()
    {
        return $this->_sendMail;
    }
    
    /**
     * Set value 'Release'
     *
     * @param string $value
     */
    public function setRelease($value)
    {
        $this->_release = $value;
    }
    
    /**
     * Ritorna il campo 'Release'
     * 
     * @return string
     */
    public function getRelease()
    {
        return $this->_release();
    }
    
    /**
     * Set value 'Options'
     *
     * @param string $value
     */
    public function setOptions($value)
    {
        $this->_options = $value;
    }
    
    /**
     * Ritorna il campo options
     * 
     * @return string
     */
    public function getOptions()
    {
        return $this->_options;
    }
    
    /**
     * Set value 'Mac'
     *
     * @param string $value
     */
    public function setMac($value)
    {
        $this->_mac = $value;
    }
    
    /**
     * Ritorna il campo 'Mac'
     * 
     * @return string
     */
    public function getMac()
    {
        return $this->_mac;
    }
    
    /**
     * Set value 'LinkExpirationDate'
     *
     * @param string $value
     */
    public function setLinkExpirationDate($value)
    {
        $this->_linkExpirationDate = $value;
    }
    
    /**
     * Ritorna il campo 'LinkExpirationDate'
     * 
     * @return string
     */
    public function getLinkExpirationDate()
    {
        return $this->_linkExpirationDate;
    }
    
    /**
     * Set value 'LinkAmount'
     *
     * @param string $value
     */
    public function setLinkAmount($value)
    {
        $this->_linkAmount = $value;
    }
    
    /**
     * Ritorna il campo 'LinkAmount'
     * 
     * @return string
     */
    public function getLinkAmount()
    {
        return $this->_linkAmount;
    }
    
    /**
     * Set value 'LinkCurrency'
     *
     * @param string $value
     */
    public function setLinkCurrency($value)
    {
        $this->_linkCurrency = $value;
    }
    
    /**
     * Ritorna il campo 'LinkCurrency'
     * 
     * @return string
     */
    public function getLinkCurrency()
    {
        return $this->_linkCurrency;
    }
    
    /**
     * Set value 'LinkExponent'
     *
     * @param string $value
     */
    public function setLinkExponent($value)
    {
        $this->_linkExponent = $value;
    }
    
    /**
     * Ritorna il campo 'LinkExponent'
     * 
     * @return string
     */
    public function getLinkExponent()
    {
        return $this->_linkExponent;
    }
    
    /**
     * Set value 'LinkOrderId'
     *
     * @param string $value
     */
    public function setLinkOrderId($value)
    {
        $this->_linkOrderId = $value;
    }
    
    /**
     * Ritorna il campo 'LinkOrderId'
     * 
     * @return string
     */
    public function getLinkOrderId()
    {
        return $this->_linkOrderId;
    }
    
    /**
     * Set value 'LinkUrlDone'
     *
     * @param string $value
     */
    public function setLinkUrlDone($value)
    {
        $this->_linkUrlDone = $value;
    }
    
    /**
     * Ritorna il campo 'LinkUrlDone'
     * 
     * @return string
     */
    public function getLinkUrlDone()
    {
        return $this->_linkUrlDone;
    }
    
    /**
     * Set value 'LinkUrlMs'
     *
     * @param string $value
     */
    public function setLinkUrlMs($value)
    {
        $this->_linkUrlMs = $value;
    }
    
    /**
     * Ritorna il campo 'LinkUrlMs'
     * 
     * @return string
     */
    public function getLinkUrlMs()
    {
        return $this->_linkUrlMs;
    }
    
    /**
     * Set value 'LinkAccountingMode'
     *
     * @param string $value
     */
    public function setLinkAccountingMode($value) 
    {
        $this->_linkAccountingMode = $value;
    }
    
    /**
     * Ritorna il campo 'LinkAccountingMode'
     * 
     * @return string
     */
    public function getLinkAccountingMode()
    {
        return $this->_linkAccountingMode;
    }
    
    /**
     * Set value 'LinkAuthorMode'
     *
     * @param string $value
     */
    public function setLinkAuthorMode($value)
    {
        $this->_linkAuthorMode = $value;
    }
    
    /**
     * Ritorna il campo 'LinkAuthorMode'
     * 
     * @return string
     */
    public function getLinkAuthorMode()
    {
        return $this->_linkAuthorMode;
    }
    
    /**
     * Set value 'LinkLang'
     *
     * @param string $value
     */
    public function setLinkLang($value)
    {
        $this->_linkLang = $value;
    }
    
    /**
     * Ritorna il campo 'LinkLang'
     * 
     * @return string
     */
    public function getLinkLang()
    {
        return $this->_linkLang;
    }
    
    /**
     * Set value 'LinkShopEmail'
     *
     * @param string $value
     */
    public function setLinkShopEmail($value)
    {
        $this->_linkShopEmail = $value;
    }
    
    /**
     * Ritorna il campo 'LinkShopEmail'
     * 
     * @return string
     */
    public function getLinkShopEmail()
    {
        return $this->_linkShopEmail;
    }
    
    /**
     * Set value 'LinkOptions'
     *
     * @param string $value
     */
    public function setLinkOptions($value)
    {
        $this->_linkOptions = $value;
    }
    
    /**
     * Ritorna il campo 'LinkOptions'
     * 
     * @return string
     */
    public function getLinkOptions()
    {
        return $this->_linkOptions;
    }
    
    /**
     * Set value 'LinkCommis'
     *
     * @param string $value
     */
    public function setLinkCommis($value)
    {
        $this->_linkCommis = $value;
    }
    
    /**
     * Ritorna il campo 'LinkCommis'
     * 
     * @return string
     */
    public function getLinkCommis()
    {
        return $this->_linkCommis;
    }
    
    /**
     * Set value 'LinkEmail'
     *
     * @param string $value
     */
    public function setLinkEmail($value)
    {
        $this->_linkEmail = $value;
    }
    
    /**
     * Ritorna il campo 'LinkEmail'
     * 
     * @return string
     */
    public function getLinkEmail()
    {
        return $this->_linkEmail;
    }
    
    /**
     * Set value 'LinkName'
     *
     * @param string $value
     */
    public function setLinkName($value)
    {
        $this->_linkName = $value;
    }
    
    /**
     * Ritorna il campo 'LinkName'
     * 
     * @return string
     */
    public function getLinkName()
    {
        return $this->_linkName;
    }
    
    /**
     * Set value 'LinkSurname'
     *
     * @param string $value
     */
    public function setLinkSurname($value)
    {
        $this->_linkSurname = $value;
    }
    
    /**
     * Ritorna il campo 'LinkSurname'
     * 
     * @return string
     */
    public function getLinkSurname()
    {
        return $this->_linkSurname;
    }
    
    /**
     * Set value 'LinkOrdDescr'
     *
     * @param string $value
     */
    public function setLinkOrdDescr($value)
    {
        $this->_linkOrdDescr = $value;
    }
    
    /**
     * Ritorna il campo 'LinkOrdDescr'
     * 
     * @return string
     */
    public function getLinkOrdDescr()
    {
        return $this->_linkOrdDescr;
    }
    
    /**
     * Set value 'LinkOpDescr'
     *
     * @param string $value
     */
    public function setLinkOpDescr($value)
    {
        $this->_linkOpDescr = $value;
    }
    
    /**
     * Ritorna il campo 'LinkOpDescr'
     * 
     * @return string
     */
    public function getLinkOpDescr()
    {
        return $this->_linkOpDescr;
    }
    
    /**
     * Set value 'LinkPhoneNumber'
     *
     * @param string $value
     */
    public function setLinkPhoneNumber($value)
    {
        $this->_linkPhoneNumber = $value;
    }
    
    /**
     * Ritorna il campo 'LinkPhoneNumber'
     * 
     * @return string
     */
    public function getLinkPhoneNumber()
    {
        return $this->_linkPhoneNumber;
    }
    
    /**
     * Set value 'LinkRemainingDuration'
     *
     * @param string $value
     */
    public function setLinkRemainingDuration($value)
    {
        $this->_linkRemainingDuration = $value;
    }
    
    /**
     * Ritorna il campo 'LinkRemainingDuration'
     * 
     * @return string
     */
    public function getLinkRemainingDuration()
    {
        return $this->_linkRemainingDuration;
    }
    
    /**
     * Set value 'LinkUserId'
     *
     * @param string $value
     */
    public function setLinkUserId($value)
    {
        $this->_linkUserId = $value;
    }
    
    /**
     * Ritorna il campo 'LinkUserId'
     * 
     * 
     * @return string
     */
    public function getLinkUserId()
    {
        return $this->_linkUserId;
    }
    
    /**
     * Esegue la validazione dei campi prima dei 
     * eseguire la richiesta al webservice
     * 
     * @throws \Exception
     */
    protected function _validate()
    {
        if(empty($this->_operation)) {
            $this->_operation = 'CREATELINK';
        }
        if(empty($this->_timestamp)) {
            $this->_timestamp = $this->_date->date()->format('Y-m-d\TH:i:s.u');
            $this->_timestamp = substr($this->_timestamp, 0, -3);        
        }
        if(empty($this->_reqRefNum)) {
            throw new \Exception("[ReqRefNum] param is mandatory");
        }
        if(empty($this->_shopId)) {
            throw new \Exception("[ShopId] param is mandatory");
        }
        if(empty($this->_operatorId)) {
            throw new \Exception("[OperatorId] param is mandatory");
        }
        if(empty($this->_release)) {
            $this->_release = '02';
        }
        if(empty($this->_linkAmount)) {
            throw new \Exception("[LinkAmount] param is mandatory");
        }
        if(empty($this->_linkCurrency)) {
            throw new \Exception("[LinkCurrency] param is mandatory");
        }
        if(empty($this->_linkOrderId)) {
            throw new \Exception("[LinkOrderId] param is mandatory");
        }
        if(empty($this->_linkUrlMs)) {
            throw new \Exception("[LinkUrlMs] param is mandatory");
        }
        if(empty($this->_linkAccountingMode)) {
            throw new \Exception("[LinkAccountingMode] param is mandatory");
        }
        if(empty($this->_linkAuthorMode)) {
            $this->_linkAuthorMode = 'I';
        }
        if(empty($this->_linkLang)) {
            throw new \Exception("[LinkLang] param is mandatory");
        }
        if(empty($this->_linkName)) {
            throw new \Exception("[LinkName] param is mandatory");
        }
        if(empty($this->_linkSurname)) {
            throw new \Exception("[LinkSurname] param is mandatory");
        }
        if(empty($this->_linkOptions)) {
            $this->_linkOptions = 'GBN';
            if(!empty($this->_linkRemainingDuration)) {
                $this->_linkOptions .= 'D';
            }
        }
        if(empty($this->_macRequestKey)) {
            throw new \Exception("[MacRequestKey] param is mandatory");
        }
        if(empty($this->_requestUrl)) {
            throw new \Exception("[RequestUrl] param is mandatory");
        }
        if(empty($this->_hash)) {
            $this->_hash = \Cooder\Sia\Model\Config\Source\Hash::HASH_SHA1;
        }
        if(empty($this->_mac)) {
            $this->_mac = $this->_calculateMac();
        }
    }
    
    /**
     * Funzione che crea la stringa XML
     * per la richiesta
     * 
     * @return string
     */
    protected function _createXml()
    {
        $xml = '';
        $xml .= '<BPWXmlRequest>';
        $xml .= '<Release>' . $this->_release . '</Release>';
        $xml .= '<Request>';
        $xml .= '<Operation>' . $this->_operation . '</Operation>';
        $xml .= '<Timestamp>' . $this->_timestamp . '</Timestamp>';
        $xml .= '<Mac>' . $this->_mac . '</Mac>';
        $xml .= '<Release>' . $this->_release . '</Release>';
        $xml .= '</Request>';
        $xml .= '<Data>';
        $xml .= '<CreateLinkRequest>';
        $xml .= '<Header>';
        $xml .= '<ShopID>' . $this->_shopId . '</ShopID>';
        $xml .= '<OperatorID>' . $this->_operatorId . '</OperatorID>';
        $xml .= '<ReqRefNum>' . $this->_reqRefNum . '</ReqRefNum>';
        $xml .= '</Header>';
        if($this->_sendMail) {
            $xml .= '<Sendmail>S</Sendmail>';
        } else {
            $xml .= '<Sendmail>N</Sendmail>';
        }
        if(!empty($this->_linkExpirationDate)) {
            $xml .= '<LinkExpirationDate>' . $this->_linkExpirationDate . '</LinkExpirationDate>';
        }
        $xml .= '<LinkAmount>' . $this->_linkAmount . '</LinkAmount>';
        $xml .= '<LinkCurrency>' . $this->_linkCurrency . '</LinkCurrency>';
        if(!empty($this->_linkExponent)) {
            $xml .= '<LinkExponent>' . $this->_linkExponent. '</LinkExponent>';
        }
        $xml .= '<LinkOrderid>' . $this->_linkOrderId . '</LinkOrderid>';
        $xml .= '<LinkUrlDone>' . $this->_linkUrlDone . '</LinkUrlDone>';
        $xml .= '<LinkUrlMs>' . $this->_linkUrlMs . '</LinkUrlMs>';
        if($this->_linkAccountingMode == \Magento\Payment\Model\Method\AbstractMethod::ACTION_AUTHORIZE) {
            $xml .= '<LinkAccountingMode>D</LinkAccountingMode>';
        } else {
            $xml .= '<LinkAccountingMode>I</LinkAccountingMode>';
        }
        $xml .= '<LinkAuthorMode>' . $this->_linkAuthorMode . '</LinkAuthorMode>';
        $xml .= '<LinkLang>' . $this->_linkLang . '</LinkLang>';
        if(!empty($this->_linkShopEmail)) {
            $xml .= '<LinkShopEmail>' . $this->_linkShopEmail . '</LinkShopEmail>';
        }
        if(!empty($this->_linkOptions)) {
            $xml .= '<LinkOptions>' . $this->_linkOptions . '</LinkOptions>';
        }
        if(!empty($this->_linkCommis)) {
            $xml .= '<LinkCommis>' . $this->_linkCommis . '</LinkCommis>';
        }
        $xml .= '<LinkEmail>' . $this->_linkEmail . '</LinkEmail>';
        $xml .= '<LinkName>' . $this->_linkName . '</LinkName>';
        $xml .= '<LinkSurname>' . $this->_linkSurname . '</LinkSurname>';
        if(!empty($this->_linkOrdDescr)) {
            $xml .= '<LinkOrdDescr>' . $this->_linkOrdDescr . '</LinkOrdDescr>';
        }
        if(!empty($this->_linkOpDescr)) {
            $xml .= '<LinkOpDescr>' . $this->_linkOpDescr . '</LinkOpDescr>';
        }
        if(!empty($this->_linkPhoneNumber)) {
            $xml .= '<LinkPhoneNumber>' . $this->_linkPhoneNumber . '</LinkPhoneNumber>';
        }
        if(!empty($this->_linkRemainingDuration)) {
            $xml .= '<LinkRemainingDuration>' . $this->_linkRemainingDuration . '</LinkRemainingDuration>';
        }
        if(!empty($this->_linkUserId)) {
            $xml .= '<LinkUserID>' . $this->_linkUserId . '</LinkUserID>';
        }
        $xml .= '</CreateLinkRequest>';
        $xml .= '</Data>';
        $xml .= '</BPWXmlRequest>';
        
        $this->_log($xml);
        
        return $xml;
    }
    
    /**
     * Esegue il parse della risposta XML data dal webservice
     * 
     * @param string $xml
     * 
     * @return \Cooder\Sia\Webservice\PaymentInit\Response
     */
    protected function _parseResponse($xml)
    {
        $this->_xmlParser->loadXML($xml);
        $content = $this->_xmlParser->xmlToArray();
        
        $this->_response->reset();
        
        if(array_key_exists('BPWXmlResponse', $content) == false) {
            throw new \Exception("[BPWXmlResponse] value missing in the XML response"); 
        }
        $content = $content['BPWXmlResponse'];
        if(array_key_exists('Result', $content) == false) {
            throw new \Exception("[Result] value missing in the XML response");
        }
        $this->_response->setResult($content['Result']);
        
        if($this->_response->getResult() == self::RESPONSE_SUCCESS_CODE) {
            if(array_key_exists('Data', $content) == false) {
                throw new \Exception("[Data] value missing in the XML response");
            }
            $content = $content['Data'];
            
            if(array_key_exists('LinkCreated', $content) == false) {
                throw new \Exception("[LinkCreated] value missing in the XML response"); 
            }
            $content = $content['LinkCreated'];
            
            if(array_key_exists('CompleteLink', $content) == false || empty($content['CompleteLink'])) {
                throw new \Exception("[CompleteLink] value missing in the XML response");
            }
            $this->_response->setCompleteLink($content['CompleteLink']);
            
            if(array_key_exists('Token', $content) == false || empty($content['Token'])) {
                throw new \Exception("[Token] value missing in the XML response");
            }
            $this->_response->setToken($content['Token']);
            
            if(array_key_exists('CreationDate', $content) == false) {
                throw new \Exception("[CreationDate] value missing in the XML response");
            }
            $this->_response->setCreationDate($content['CreationDate']);
            
            if(array_key_exists('Status', $content) == false) {
                throw new \Exception("[Status] value missing in the XML response");
            }
            $this->_response->setStatus($content['Status']);
            
            if(array_key_exists('LastUseDate', $content) == false) {
                throw new \Exception("[LastUseDate] value missing in the XML response");
            }
            $this->_response->setLastUseDate($content['LastUseDate']);
            
            if(array_key_exists('ExpirationDate', $content) == false) {
                throw new \Exception("[ExpirationDate] value missing in the XML response");
            }
            $this->_response->setExpirationDate($content['ExpirationDate']);
            
            if(array_key_exists('RevokeDate', $content) == false) {
                throw new \Exception("[RevokeDate] value missing in the XML response");
            }
            $this->_response->setRevokeDate($content['RevokeDate']);
            
            if(array_key_exists('OrderId', $content) == false || empty($content['OrderId'])) {
                throw new \Exception("[OrderId] value missing in the XML response");
            }
            $this->_response->setOrderId($content['OrderId']);
            
            if(array_key_exists('MAC', $content) == false || empty($content['MAC'])) {
                throw new \Exception("[MAC] value missing in the XML response");
            }
            $this->_response->setMac($content['MAC']);
        }
        
        return $this->_response;
    }
    
    /**
     * Calcola il codice MAC dati i parametri 
     * della richiesta
     * 
     * @return string
     */
    protected function _calculateMac()
    {
        $hash = 'OPERATION='. $this->_operation . '&';
        $hash .= 'TIMESTAMP='. $this->_timestamp . '&';
        $hash .= 'REQREFNUM='. $this->_reqRefNum . '&';    
        $hash .= 'SHOPID='. $this->_shopId . '&';
        $hash .= 'OPERATORID='. $this->_operatorId . '&';
        if($this->_sendMail) {
            $hash .= 'SENDMAIL=S&';
        } else {
            $hash .= 'SENDMAIL=N&';
        }
        if(!empty($this->_linkExpirationDate)) {
            $hash .= 'LINKEXPIRATIONDATE='. $this->_linkExpirationDate . '&';
        }
        $hash .= 'LINKAMOUNT='. $this->_linkAmount . '&';
        $hash .= 'LINKCURRENCY='. $this->_linkCurrency . '&';
        if(!empty($this->_linkExponent)) {
            $hash .= 'LINKEXPONENT='. $this->_linkExponent . '&';
        }
        $hash .= 'LINKORDERID='. $this->_linkOrderId . '&';
        if(!empty($this->_linkUrlDone)) {
            $hash .= 'LINKURLDONE='. $this->_linkUrlDone . '&';
        }
        $hash .= 'LINKURLMS='. $this->_linkUrlMs . '&';
        if($this->_linkAccountingMode == \Magento\Payment\Model\Method\AbstractMethod::ACTION_AUTHORIZE) {
            $hash .= 'LINKACCOUNTINGMODE=D&';
        } else {
            $hash .= 'LINKACCOUNTINGMODE=I&';
        }
        $hash .= 'LINKAUTHORMODE='. $this->_linkAuthorMode . '&';
        $hash .= 'LINKLANG='. $this->_linkLang . '&';
        if(!empty($this->_linkShopEmail)) {
            $hash .= 'LINKSHOPEMAIL='. $this->_linkShopEmail . '&';
        }
        if(!empty($this->_linkOptions)) {
            $hash .= 'LINKOPTIONS='. $this->_linkOptions . '&';
        }
        if(!empty($this->_linkCommis)) {
            $hash .= 'LINKCOMMIS='. $this->_linkCommis . '&';
        }
        $hash .= 'LINKEMAIL='. $this->_linkEmail . '&';
        $hash .= 'LINKNAME='. $this->_linkName . '&';
        $hash .= 'LINKSURNAME='. $this->_linkSurname . '&';
        if(!empty($this->_linkOrdDescr)) {
            $hash .= 'LINKORDDESCR='. $this->_linkOrdDescr . '&';
        }
        if(!empty($this->_linkOpDescr)) {
            $hash .= 'LINKOPDESCR='. $this->_linkOpDescr . '&';
        }
        if(!empty($this->_linkPhoneNumber)) {
            $hash .= 'LINKPHONENUMBER='. $this->_linkPhoneNumber . '&';
        }
        if(!empty($this->_linkRemainingDuration)) {
            $hash .= 'LINKREMAININGDURATION='. $this->_linkRemainingDuration . '&';
        }
        if(!empty($this->_linkUserId)) {
            $hash .= 'LINKUSERID='. $this->_linkUserId . '&';
        }
        if(!empty($this->_options)) {
            $hash .= 'OPTIONS='. $this->_options . '&';
        }
        
        if($this->_hash == \Cooder\Sia\Model\Config\Source\Hash::HASH_SHA1) {
            $hash .= $this->_macRequestKey;
            return sha1($hash);
        } elseif($this->_hash == \Cooder\Sia\Model\Config\Source\Hash::HASH_MD5) {
            $hash .= $this->_macRequestKey;
            return md5($hash);
        } else {
            return hash_hmac('sha256', $hash, $this->_macRequestKey);
        }
    }
    
    /**
     * Invia la richiesta al webservice per la creazione del link di pagamento
     * e restituisce la response
     *
     * @return \Cooder\Sia\Webservice\PaymentInit\Response
     */
    public function send()
    {
        try {
            $this->_validate();
            $xml = $this->_createXml();
            
            $this->_curl->post($this->_requestUrl, ['data' => $xml]);
            $response = $this->_curl->getBody();
            
            $this->_log($response);
            
            return $this->_parseResponse($response);
        } catch(\Exception $ex) {
            throw new \Exception("CreateLink Request failed: " . $ex->getMessage());   
        }
    }
    
    /**
     * Scrive log se attivato
     * 
     * @param string $message
     * @param boolean $forceLog
     */
    protected function _log($message, $forceLog = false)
    {
        if ($this->_baseLogger && ($forceLog || $this->_debug)) {
            $this->_baseLogger->debug($message);
        }
    }
}