<?php

namespace Cooder\Sia\Model;

/**
 * Metodo di pagamento Cooder_SiaPay
 * 
 * @author Devid Marcantoni <devid@cooder.it>
 */
class SiaPay extends \Magento\Payment\Model\Method\AbstractMethod
{

    const PAYMENT_URL_ADDITIONAL_INFO_KEY   = 'sia_payment_url';
    const PAYMENT_TYPE                      = 'payment_type';
    const PAYMENT_TOKEN                     = 'sia_payment_token';
    const PAYMENT_SIA_TYPE                  = 'sia_payment_sya_type';
    
    const CURRENCY_CODE_EUR = '978';
    const CURRENCY_CODE_USD = '840';
    const CURRENCY_EXPONENT_USD = '2';
    
    /**
     * Payment code
     *
     * @var string
     */
    protected $_code = 'sia';
    
    /**
     * Oggetto per la request al webservice
     * per la creazione del link di pagamento
     * 
     * @var \Cooder\Sia\Model\Webservice\PaymentInit\Request
     */
    protected $_webserviceRequest;
    
    /**
     * Date Object
     *
     * @var \Magento\Framework\Stdlib\DateTime\TimezoneInterface
     */
    protected $_date;
    
    /**
     * Directory Helper
     * 
     * @var \Magento\Directory\Helper\Data
     */
    protected $_directoryHelper;
    
    /**
     * Price Currency
     * 
     * @var \Magento\Framework\Pricing\PriceCurrencyInterface
     */
    protected $_priceCurrency;
    
    /**
     * Store Manager
     * 
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $_storeManager;
    
    /**
     * Order Sender
     * 
     * @var \Magento\Sales\Model\Order\Email\Sender\OrderSender
     */
    protected $_orderSender;
    
    /**
     * Order Repository
     * 
     * @var \Magento\Sales\Api\OrderRepositoryInterface
     */
    protected $_orderRepository;
    
    /**
     * Factory per il recupero dei dati sugli ordini
     *
     * @var \Magento\Sales\Model\OrderFactory
     */
    protected $_orderFactory;
    
    /**
     * Checkout Session
     *
     * @var \Magento\Checkout\Model\Session
     */
    protected $_checkoutSession;
    
    /**
     * Logger del modulo
     * 
     * @var \Cooder\Sia\Logger\Logger
     */
    protected $_baseLogger;
    
    /**
     * Helper Data
     * 
     * @var \Cooder\Sia\Helper\Data
     */
    protected $_helper;
    
   /**
    * Costruttore
    * 
    * @param \Magento\Framework\Model\Context $context
    * @param \Magento\Framework\Registry $registry
    * @param \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory
    * @param \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory
    * @param \Magento\Payment\Helper\Data $paymentData
    * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
    * @param \Magento\Payment\Model\Method\Logger $logger
    * @param \Cooder\Sia\Model\Webservice\PaymentInit\Request $webserviceRequest
    * @param \Magento\Framework\Stdlib\DateTime\TimezoneInterface $date
    * @param \Magento\Directory\Helper\Data $directoryHelper
    * @param \Magento\Framework\Pricing\PriceCurrencyInterface $priceCurrency
    * @param \Magento\Store\Model\StoreManagerInterface $storeManager
    * @param \Magento\Sales\Model\Order\Email\Sender\OrderSender $orderSender
    * @param \Magento\Sales\Api\OrderRepositoryInterface $orderRepository
    * @param \Magento\Sales\Model\OrderFactory $orderFactory
    * @param \Magento\Checkout\Model\Session $checkoutSession
    * @param \Cooder\Sia\Logger\Logger $baseLogger
    * @param \Magento\Framework\Model\ResourceModel\AbstractResource $resource
    * @param \Magento\Framework\Data\Collection\AbstractDb $resourceCollection
    * @param array $data
    */
    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Cooder\Sia\Model\Webservice\PaymentInit\Request $webserviceRequest,
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $date,
        \Magento\Directory\Helper\Data $directoryHelper,
        \Magento\Framework\Pricing\PriceCurrencyInterface $priceCurrency,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Sales\Model\Order\Email\Sender\OrderSender $orderSender,
        \Magento\Sales\Api\OrderRepositoryInterface $orderRepository,
        \Magento\Sales\Model\OrderFactory $orderFactory,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Cooder\Sia\Logger\Logger $baseLogger,
        \Cooder\Sia\Helper\Data $helper,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = []
        ) {
            parent::__construct(
                $context,
                $registry,
                $extensionFactory,
                $customAttributeFactory,
                $paymentData,
                $scopeConfig,
                $logger,
                $resource,
                $resourceCollection,
                $data
            );
            $this->_webserviceRequest = $webserviceRequest;
            $this->_date = $date;
            $this->_directoryHelper = $directoryHelper;
            $this->_priceCurrency = $priceCurrency;
            $this->_storeManager = $storeManager;
            $this->_orderSender = $orderSender;
            $this->_orderRepository = $orderRepository;
            $this->_orderFactory = $orderFactory;
            $this->_checkoutSession = $checkoutSession;
            $this->_baseLogger = $baseLogger;
            $this->_helper = $helper;
            
            $this->_isInitializeNeeded = true;
            $this->_canUseInternal = false;
    }
    
    /**
     * {@inheritdoc}
     */
    public function initialize($paymentAction, $stateObject)
    {
        $this->_log(sprintf('SiaPay - Called payment initialize. Action is %s', $paymentAction), true);
        switch ($paymentAction) {
            case self::ACTION_AUTHORIZE:
            case self::ACTION_AUTHORIZE_CAPTURE:
                
                // Recupero payment e order
                $payment = $this->getInfoInstance();
                $order = $payment->getOrder();
                $order->setCanSendNewEmailFlag(false);
                $this->_log(sprintf('SiaPay - Retrieved order with increment id %s', $order->getIncrementId()), true);
                                
                // Imposto amount da autorizzare
                $payment->setAmountAuthorized($order->getTotalDue());
                $payment->setBaseAmountAuthorized($order->getBaseTotalDue());
                
                // Definisco i dati di redirect
                $redirectUrl = $this->_storeManager->getStore()->getBaseUrl() . 'sia/redirect/form';
                $token = '';
                
                $stateObject->setState(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT);
                $stateObject->setStatus($order->getConfig()->getStateDefaultStatus(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT));
                $stateObject->setIsNotified(false);
                $this->_setPaymentAdditionalInformation($payment, self::PAYMENT_URL_ADDITIONAL_INFO_KEY, $redirectUrl);
                $this->_setPaymentAdditionalInformation($payment, self::PAYMENT_TYPE, $paymentAction);
                $this->_setPaymentAdditionalInformation($payment, self::PAYMENT_TOKEN, $token);
                $this->_setPaymentAdditionalInformation($payment, self::PAYMENT_SIA_TYPE, $this->getConfigData('payment_type'));
                break;
            default:
                $message = sprintf('Payment method "%s" does not support "%s" payment action.', $this->getCode(), $paymentAction);
                throw new \InvalidArgumentException($message);
        }
        $this->_log('SiaPay - Payment initialization complete.', true);
        return $this;
    }
    
    /**
     * Verifica il risultato del pagamento dati i parametri 
     * dal server Sia
     * 
     * @param Array<mixed> $params
     * @param boolean $isS2S
     * 
     * @throws \Exception
     * @return boolean
     */
    public function paymentVerify($params, $isS2S = false)
    {
        // Verifico parametro ordine
        if(array_key_exists('ORDERID', $params) == false) {
            $this->_log("[ORDERID] param missing", true);
            return false;
        }
        $orderIncrementId = $params['ORDERID'];
        
        $this->_log(sprintf('Currently verifying payment for order "#%s"...', $orderIncrementId), true);
        
        // Istanzio ordine 
        try {
            $order = $this->_orderFactory->create()->loadByIncrementId($orderIncrementId);
            if(!$order->getId()) {
                $this->_log("No order for param [ORDERID]", true);
                return false;
            }
        } catch(\Exception $ex) {
            $this->_log("No order for param [ORDERID]", true);
            return false;
        }
        
        // Recupero ordine in sessione
        if($isS2S == false) {
            $sessionOrder = $this->_checkoutSession->getLastRealOrder();
            if($order->getIncrementId() != $sessionOrder->getIncrementId()) {
                $this->_log("Current order [" . $order->getIncrementId() . "] is not the session order [" . $sessionOrder->getIncrementId() . "]", true);
                $this->_setPaymentError($order, 'XX', __('Current order in not the session order'));
                return false;
            }
        }
       
        // Verifico parametro risultato
        if(array_key_exists('RESULT', $params) == false) {
            $this->_log("[RESULT] param missing", true);
            $this->_setPaymentError($order, 'XX', __('[Result] param missing'));
            return false;
        }
        $result = $params['RESULT'];
        
        $this->_log(sprintf('Payment result code "#%s"', $result), true);
        
        // Verifico se ho risposta negativa e, nel caso, cancello ordine
        if($result != \Cooder\Sia\Model\Webservice\PaymentInit\Request::RESPONSE_SUCCESS_CODE) {
            $errorMessage = $this->_getPaymentErrorMessage($result);
            $this->_log(sprintf('Transaction result for order %s is negative', $order->getIncrementId()), true);
            $this->_setPaymentError($order, $result, $errorMessage);
            return false;
        } 
        
        $this->_log(sprintf('Transaction result for order %s is positive', $order->getIncrementId()), true);
        
        // Verifico identificatore transazione
        if(array_key_exists('TRANSACTIONID', $params) == false || empty($params['TRANSACTIONID'])) {
            $this->_log("[TRANSACTIONID] param missing", true);
            $this->_setPaymentError($order, 'XX', __('[TRANSACTIONID] param missing'));
            return false;
        }
        $transactionId = $params['TRANSACTIONID'];
        
        // Verifico authorization code
        $authCode = '';
        if(array_key_exists('AUTHNUMBER', $params)) {
            $authCode = $params['AUTHNUMBER'];
        }
        
        // Verifico esistenza codice MAC
        if(array_key_exists('MAC', $params) == false) {
            $this->_log("[MAC] param missing", true);
            $this->_setPaymentReview($order, '[MAC] param missing');
            return false;
        }
        $mac = $params['MAC'];
        
        // Verifico correttezza codice MAC
        if($this->_verifyMac($params, $mac) == false) {
            $this->_log("[MAC] param verify fail", true);
            $this->_setPaymentReview($order, '[MAC] param verify fail');
            return false;
        }
        
        // Salvo gli estremi di pagamento
        $payment = $order->getPayment();
        $this->_importInfoToPayment($payment, $transactionId, $authCode);
                
        $this->_log(sprintf('Payment action is "%s"', $this->_getPaymentAction($payment)), true);
        
        if ($this->_isAuthorizeCapturePaymentAction($payment)) {
            $this->_handleNotificationAsCapture($payment);
            $this->_log(sprintf('Registered capture notification for order "%s"', $order->getIncrementId()), true);
        } elseif ($this->_isOnlyAuthorizePaymentAction($payment)) {
            $this->_handleNotificationAsAuthorization($payment);
            $this->_log(sprintf('Registered authorization notification for order "%s"', $order->getIncrementId()), true);
        } else {
            $message = sprintf(
                'Invalid payment action "%s" for payment method "%s". Allowed payment actions are %s.',
                $this->getPaymentAction($payment),
                $this->getCode(),
                implode(', ', [self::ACTION_AUTHORIZE_CAPTURE, self::ACTION_AUTHORIZE])
            );
            $this->_log($message, true);
            throw new \RuntimeException($message);
        }
        
        $this->_log(sprintf('Verify for order %s complete', $order->getIncrementId()), true);
        return true;
    }
    
    /**
     * Handle per avvenuta autorizzazione pagamento
     * 
     * @param \Magento\Sales\Api\Data\OrderPaymentInterface $payment
     */
    protected function _handleNotificationAsAuthorization(\Magento\Sales\Api\Data\OrderPaymentInterface $payment)
    {
        /** @var \Magento\Sales\Model\Order\Payment $payment */
        /** @var \Magento\Sales\Model\Order $order */
        $order = $payment->getOrder();
        
        $this->_log('Saving authorization for order '. $order->getIncrementId(), true);
        
        $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING);
        $payment->registerAuthorizationNotification($payment->getBaseAmountAuthorized());
        $this->_orderRepository->save($order);
        
        if (!$order->getEmailSent()) {
            $this->_log('Sending email to customer for order ' . $order->getIncrementId(), true);
            $this->_orderSender->send($order);
            /** @var \Magento\Sales\Api\Data\OrderStatusHistoryInterface $comment */
            $comment = $order->addStatusHistoryComment(__('Notified customer about payment authorization.'));
            $comment->setIsCustomerNotified(true);
            $this->_orderRepository->save($order);
        }
    }
    
    /**
     * Handle per avvenuto capture pagamento
     * 
     * @param \Magento\Sales\Api\Data\OrderPaymentInterface
     */
    protected function _handleNotificationAsCapture(\Magento\Sales\Api\Data\OrderPaymentInterface $payment)
    {
        /** @var \Magento\Sales\Model\Order\Payment $payment */
        /** @var \Magento\Sales\Model\Order $order */
        $order = $payment->getOrder();
        
        $this->_log('Saving capture for order '. $order->getIncrementId(), true);
        
        $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING);
        $payment->registerCaptureNotification($payment->getBaseAmountAuthorized());
        $this->_orderRepository->save($order);
        
        /** @var \Magento\Sales\Api\Data\InvoiceInterface $invoice */
        $invoice = $payment->getCreatedInvoice();
        if ($invoice && !$order->getEmailSent()) {
            $this->_log('Sending email to customer for order ' . $order->getIncrementId(), true);
            $this->_orderSender->send($order);
            /** @var \Magento\Sales\Api\Data\OrderStatusHistoryInterface $comment */
            $comment = $order->addStatusHistoryComment(__('Notified customer about invoice #%1.', $invoice->getIncrementId()));
            $comment->setIsCustomerNotified(true);
            $this->_orderRepository->save($order);
        }
    }
    
    /**
     * Setta il pagamento con esito negativo.
     * Se necessario esegue cancellazione ordine
     * 
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @param string $errorCode
     * @param string $errorMessage
     */
    protected function _setPaymentError(\Magento\Sales\Api\Data\OrderInterface $order, $errorCode, $errorMessage)
    {
        /** @var \Magento\Sales\Model\Order $order */
        /** @var \Magento\Sales\Api\Data\OrderStatusHistoryInterface $comment */
        $comment = $order->addStatusHistoryComment(
            sprintf(
                'SiaPay transaction result is negative (Rc="%s", ErrorDesc="%s")',
                $errorCode,
                $errorMessage
            )
        );
        $comment->setIsCustomerNotified(false);
        $this->_orderRepository->save($order);
        
        if ($this->getConfigData('cancel_on_failure') && $order->canCancel()) {
            $this->_log(sprintf('Canceling order %s, as indicated in the module configuration', $order->getIncrementId()), true);
            $order->cancel();
            $comment = $order->addStatusHistoryComment(__('Order automatically canceled upon reception of an error from the payment gateway.'));
            $comment->setIsCustomerNotified(false);
            $this->_orderRepository->save($order);
        }
    }
    
    /**
     * Setta ordine con lo stato PAYMENT_REVIEW in quanto 
     * presente un errore sui dati
     *
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @param string $errorMessage
     */
    protected function _setPaymentReview(\Magento\Sales\Api\Data\OrderInterface $order, $errorMessage)
    {
        /** @var \Magento\Sales\Model\Order $order */
        /** @var \Magento\Sales\Api\Data\OrderStatusHistoryInterface $comment */
        $comment = $order->addStatusHistoryComment(
            sprintf(
                'SiaPay transaction has errors (ErrorDesc="%s")',
                $errorMessage
            )
        );
        $comment->setIsCustomerNotified(false);
        $this->_orderRepository->save($order);
        
        $order->setState(\Magento\Sales\Model\Order::STATE_PAYMENT_REVIEW);
        $order->setStatus(\Magento\Sales\Model\Order::STATE_PAYMENT_REVIEW);
        $this->_orderRepository->save($order);
    }
    
    /**
     * Salva gli estremi sulla transazione e chiude il pagamento
     * 
     * @param \Magento\Sales\Api\Data\OrderPaymentInterface $payment
     * @param string $transactionId
     * @param string $authCode
     */
    protected function _importInfoToPayment(\Magento\Sales\Api\Data\OrderPaymentInterface $payment, $transactionId, $authCode)
    {
        /** @var \Magento\Sales\Model\Order\Payment $payment */
        $payment->setTransactionId($transactionId);
        $payment->setCurrencyCode($payment->getOrder()->getBaseCurrencyCode());
        $payment->setPreparedMessage(
            sprintf(
                'SiaPay successful payment result message received (AuthCode="%s").',
                $authCode
            )
        );
        $payment->setIsTransactionClosed(0);
    }
    
    /**
     * Imposta informazioni al payment
     * 
     * @param \Magento\Sales\Api\Data\OrderPaymentInterface $payment
     * @param string $key
     * @param mixed $value
     */
    protected function _setPaymentAdditionalInformation(\Magento\Sales\Api\Data\OrderPaymentInterface $payment, $key, $value)
    {
        /** @var \Magento\Sales\Model\Order\Payment $payment */
        $payment->setAdditionalInformation($key, $value);
    }
    
    /**
     * Restituisce il grand total ordine con eventuale
     * conversione di valuta
     * 
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @return double
     */
    protected function _getPaymentPrice(\Magento\Sales\Api\Data\OrderInterface $order)
    {
        return $this->_priceCurrency->round(
            $this->_directoryHelper->currencyConvert(
                $order->getBaseGrandTotal(),
                $order->getBaseCurrencyCode(),
                $this->_currencyCodeConversion($this->getConfigData('currency'))
            )
        ) * 100;
    }
    
    /**
     * Calcola il valore per il campo ReqRefNum
     * 
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @return string
     */
    protected function _calculateReqRefNum(\Magento\Sales\Api\Data\OrderInterface $order)
    {
        return $this->_date->date()->format('Ymd') . str_pad(time(), 24, '0', STR_PAD_LEFT);
    }
    
    /**
	 * Conversioni tra codici di valuta
	 *
	 * @param string $currencyCode
	 * @return string
	 */
	protected function _currencyCodeConversion($currencyCode)
	{
	    switch($currencyCode) {
	        case self::CURRENCY_CODE_EUR: {
	            return 'EUR';
	            break;
	        }
	        case self::CURRENCY_CODE_USD: {
	            return 'USD';
	            break;
	        }
	    }
	}
    
    /**
     * Scrive log se attivato
     * 
     * @param string $message
     * @param boolean $forceLog
     */
    protected function _log($message, $forceLog = false)
    {
        if ($this->_baseLogger && ($forceLog || $this->getDebugFlag())) {
            $this->_baseLogger->debug($message);
        }
    }
    
    /**
     * Ritorna la action di un payment
     * 
     * @param OrderPaymentInterface $payment
     * @return string
     */
    protected function _getPaymentAction(\Magento\Sales\Api\Data\OrderPaymentInterface $payment)
    {
        $additionalInformation = $payment->getAdditionalInformation();
        if (is_array($additionalInformation) && array_key_exists(self::PAYMENT_TYPE, $additionalInformation)) {
            $infoPaymentAction = $additionalInformation[self::PAYMENT_TYPE];
            if ($infoPaymentAction) {
                return $infoPaymentAction;
            }
        }
        
        $this->_log('Missing payment type data on payment', true);
        return $this->getConfigPaymentAction();
    }
    
    /**
     * Ritorna se il tipo di pagamento è capture
     * 
     * @param \Magento\Sales\Api\Data\OrderPaymentInterface $payment
     * @return bool
     */
    protected function _isAuthorizeCapturePaymentAction(\Magento\Sales\Api\Data\OrderPaymentInterface $payment)
    {
        return $this->_getPaymentAction($payment) === self::ACTION_AUTHORIZE_CAPTURE;
    }
    
    /**
     * Ritorna se il tipo di pagamento è solo authorize
     * 
     * @param \Magento\Sales\Api\Data\OrderPaymentInterface $payment
     * @return bool
     */
    protected function _isOnlyAuthorizePaymentAction(\Magento\Sales\Api\Data\OrderPaymentInterface $payment)
    {
        return $this->_getPaymentAction($payment) === self::ACTION_AUTHORIZE;
    }
    
    /**
     * Ritorna messaggio di errore mancato pagamento
     * dato il codice errore
     * 
     * @param string $errorCode
     * @return string
     */
    protected function _getPaymentErrorMessage($errorCode) 
    {
        switch($errorCode) {
            case "01": {
                return __("Denied by system");
                break;
            }
            case "02": {
                return __("Denied due to store configuration issues");
                break;
            }
            case "03": {
                return __("Denied due to communication issues with the authorization circuits");
                break;
            }
            case "04": {
                return __("Denied by card issuer");
                break;
            }
            case "05": {
                return __("Denied due to incorrect card number");
                break;
            }
            case "06": {
                return __("Unforeseen error during processing of request ");
                break;
            }
            case "07": {
                return __("Duplicated order ");
                break;
            }
            case "60": {
                return __("Denied due to failed Riskshield antifraud check");
                break;
            }
            case "61": {
                return __("Denied due to failed antifraud check AmexPan");
                break;
            }
            case "62": {
                return __("Denied due to failed antifraud check AmexPanIP");
                break;
            }
            case "63": {
                return __("Denied due to failed antifraud check H3GPan");
                break;
            }
            case "64": {
                return __("Denied due to failed antifraud check ItaPanCountry");
                break;
            }
            case "65": {
                return __("Denied due to failed antifraud check PaypalCountry");
                break;
            }
            case "66": {
                return __("Denied due to failed antifraud check CardEnrolledAuthenticate");
                break;
            }
            case "67": {
                return __("Denied due to failed antifraud check PanBlackList");
                break;
            }
            case "68": {
                return __("Denied due to failed antifraud check CountryPan");
                break;
            }
            case "69": {
                return __("Denied due to failed antifraud check PrepaidPan");
                break;
            }
            default: {
                return __("Unknown error");
                break;
            }
        }
    }
    
    /**
     * Verifica la correttezza del codice MAC
     * 
     * @param Array<mixed> $params
     * @param string $mac
     * 
     * @return boolean
     */
    protected function _verifyMac($params, $mac)
    {
        $sortedKeys = [
            'ORDERID',
            'SHOPID',
            'AUTHNUMBER',
            'AMOUNT',
            'CURRENCY',
            'EXPONENT',
            'TRANSACTIONID',
            'ACCOUNTINGMODE',
            'AUTHORMODE',
            'RESULT',
            'TRANSACTIONTYPE',
            'ISSUERCOUNTRY',
            'AUTHCODE',
            'PAYERID',
            'PAYER',
            'PAYERSTATUS',
            'HASHPAN',
            'PANALIASREV',
            'PANALIAS',
            'PANALIASEXPDATE',
            'PANALIASTAIL',
            'MASKEDPAN',
            'PANTAIL',
            'PANEXPIRYDATE',
            'ACCOUNTHOLDER',
            'IBAN',
            'ALIASSTR',
            'ACQUIRERBIN',
            'MERCHANTID',
            'CARDTYPE'
        ];
        
        $strToHash = '';
        foreach ($sortedKeys as $key) {
            if(array_key_exists($key, $params)) {
                $strToHash .= $key . '=' . $params[$key] . '&';
            }
        }
        
        $hash = $this->getConfigData('hash');
        if($hash == \Cooder\Sia\Model\Config\Source\Hash::HASH_SHA1) {
            $strToHash .= $this->_helper->getMacApiKey();
            return sha1($strToHash);
        } elseif($hash == \Cooder\Sia\Model\Config\Source\Hash::HASH_MD5) {
            $strToHash .= $this->_helper->getMacApiKey();
            return md5($strToHash);
        } else {
            return hash_hmac('sha256', $strToHash, $this->_helper->getMacApiKey());
        }
    }
    
    /**
     * {@inheritDoc}
     * @see \Magento\Payment\Model\Method\AbstractMethod::cancel()
     */
    public function cancelOperation(\Magento\Sales\Api\Data\OrderInterface $order)
    {
        if ($order->canCancel()) {
            $this->_log(sprintf('Canceling order %s, as indicated in the module configuration', $order->getIncrementId()), true);
            $order->cancel();
            $comment = $order->addStatusHistoryComment(__('Order canceled by user request.'));
            $comment->setIsCustomerNotified(false);
            $this->_orderRepository->save($order);
        }
    }
    
    /**
     * {@inheritDoc}
     * @see \Magento\Payment\Model\Method\AbstractMethod::isActive()
     */
    public function isActive($storeId = null)
    {
        return (bool)(int)$this->getConfigData('active_group/active', $storeId);
    }
    
    /**
     * {@inheritDoc}
     * @see \Magento\Payment\Model\Method\AbstractMethod::getConfigData()
     */
    public function getConfigData($field, $storeId = null)
    {
        if ('active' === $field) {
            if (null === $storeId) {
                $storeId = $this->getStore();
            }
            $path = 'payment/' . $this->getCode() . '/active_group/' . $field;
            return $this->_scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
        } elseif('payment_action' === $field) {
            if (null === $storeId) {
                $storeId = $this->getStore();
            }
            $path = 'payment/' . $this->getCode() . '/payment_action_group/' . $field;
            return $this->_scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
        }
        return parent::getConfigData($field, $storeId);
    }
}