<?php
/**
 * BSS Commerce Co.
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the EULA
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at thisURL:
 * http://bsscommerce.com/Bss-Commerce-License.txt
 *
 * @category   BSS
 * @package    Bss_SizeChart
 * @author     Extension Team
 * @copyright  Copyright (c) 2017-2018 BSS Commerce Co. ( http://bsscommerce.com )
 * @license    http://bsscommerce.com/Bss-Commerce-License.txt
 */
namespace Bss\SizeChart\Controller\Adminhtml\SizeChart;

use Magento\Backend\App\Action;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Eav\Model\ResourceModel\Entity\Attribute;
use Bss\SizeChart\Model\SizeChartFactory;
use Magento\Backend\Model\SessionFactory;
use Bss\SizeChart\Model\ResourceModel\SizeChart;

class Save extends \Magento\Backend\App\Action
{

    /**
     * ProductFactory
     *
     * @var ProductFactory
     */
    protected $productFactory;

    /**
     * CollectionFactory
     *
     * @var CollectionFactory
     */
    protected $collectionProductFactory;

    /**
     * ResourceConnection
     *
     * @var ResourceConnection
     */
    protected $resource;

    /**
     * ScopeConfigInterface
     *
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * Attribute
     *
     * @var Attribute
     */
    protected $eavAttribute;

    /**
     * SizeChartFactory
     *
     * @var SizeChartFactory
     */
    protected $sizeChartFactory;

    /**
     * SessionFactory
     *
     * @var SessionFactory
     */
    protected $sessionFactory;

    /**
     * SizeChart
     *
     * @var SizeChart
     */
    protected $resourceSizeChart;

    protected $resourceRule;

    protected $ruleFactory;

    /**
     * Core Registry
     * @var \Magento\Framework\Registry
     */
    protected $coreRegistry;

    /**
     * Save constructor.
     * @param CollectionFactory $collectionProductFactory
     * @param Action\Context $context
     * @param ScopeConfigInterface $scopeConfig
     * @param ProductFactory $productFactory
     * @param ResourceConnection $resource
     * @param Attribute $eavAttribute
     * @param SizeChartFactory $sizeChartFactory
     * @param SessionFactory $sessionFactory
     * @param SizeChart $resourceSizeChart
     */
    public function __construct(
        CollectionFactory $collectionProductFactory,
        Action\Context $context,
        ScopeConfigInterface $scopeConfig,
        ProductFactory $productFactory,
        ResourceConnection $resource,
        Attribute $eavAttribute,
        SizeChartFactory $sizeChartFactory,
        SessionFactory $sessionFactory,
        SizeChart $resourceSizeChart,
        \Bss\SizeChart\Model\ResourceModel\Rule $resourceRule,
        \Bss\SizeChart\Model\RuleFactory $ruleFactory,
        \Magento\Framework\Registry $coreRegistry
    ) {
        parent::__construct($context);
        $this->collectionProductFactory = $collectionProductFactory;
        $this->productFactory = $productFactory;
        $this->scopeConfig = $scopeConfig;
        $this->resource = $resource;
        $this->eavAttribute = $eavAttribute;
        $this->sizeChartFactory = $sizeChartFactory;
        $this->sessionFactory = $sessionFactory;
        $this->resourceSizeChart = $resourceSizeChart;
        $this->resourceRule = $resourceRule;
        $this->ruleFactory = $ruleFactory;
        $this->coreRegistry = $coreRegistry;
    }

    /**
     * Get default display
     * @return int
     */
    protected function getDefaultDisplay()
    {
        $displayDefault = $this->scopeConfig->getValue(
            'bss_sizechart/bss_sizechart_setting/default_display_bss_sizechart',
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );
        $displayDefaultReturn = isset($displayDefault) ? $displayDefault : 1;
        return $displayDefaultReturn;
    }

    /**
     * Get Array product Id
     * @param int $arrProductId
     * @return int array
     */
    protected function deleteOnArray($arrProductId)
    {
        if (end($arrProductId) == "on") {
            array_pop($arrProductId);
            return $arrProductId;
        } else {
            return $arrProductId;
        }
    }

    /**
     * Convert String to Array
     * @param string $string
     * @return array
     */
    protected function stringToArrayProductId($string)
    {
        $arrProductId = explode("&", $string);
        $this->deleteOnArray($arrProductId);
        return $arrProductId;
    }

    /**
     * Get Array Product By Size Chart
     * @param int $idSizeChart
     * @return array
     */
    protected function getArrProductBySizeChart($idSizeChart)
    {
        $attributeId = $this->eavAttribute->getIdByCode(
            'catalog_product',
            'bss_sizechart'
        );
        $arrProductId = $this->resourceSizeChart->getArrProductBySizeChart($idSizeChart, $attributeId);
        $arrProductIdChance = [];
        foreach ($arrProductId as $productId) {
            array_push($arrProductIdChance, $productId["entity_id"]);
        }
        return $arrProductIdChance;
    }

    /**
     * Get Array Product
     * @param array $arrayFirst
     * @param array $arraySecond
     * @return array
     */
    protected function getProducts($arrayFirst = [], $arraySecond = [])
    {
        $productsSelected = array_diff($arrayFirst, $arraySecond);
        return $productsSelected;
    }

    /**
     * Get Size Chart By Id
     * @param int $idSizeChart
     * @return \Bss\SizeChart\Model\SizeChart
     */
    protected function getSizeChartById($idSizeChart)
    {
        $sizeChart = $this->sizeChartFactory->create()->load($idSizeChart);
        return $sizeChart;
    }

    /**
     * Update Products
     * @param array $arrayProductsId
     * @param int $sizeChartId
     * @param int $overwrite
     * @param int $storeId
     * @return void
     */
    protected function updateProducts($arrayProductsId = [], $sizeChartId, $overwrite,$storeId)
    {
        $attributeId = $this->eavAttribute->getIdByCode(
            'catalog_product',
            'bss_sizechart'
        );
        foreach ($arrayProductsId as $productId) {
            $connection = $this->resource->getConnection();
            $table = $connection->getTableName('catalog_product_entity_varchar');
            $sql = $this->resourceSizeChart->buildSql($table, $productId, $attributeId, $storeId);
            $result = $connection->fetchRow($sql);
            if ($result != null) {
                if ($sizeChartId != "0") {
                    $sizeChart = $this->getSizeChartById($result);
                    $sizeChartAfter = $this->getSizeChartById($sizeChartId);
                    if ($overwrite == 0) {
                        if ($sizeChart->getPriority() <= $sizeChartAfter->getPriority() && isset($sizeChartAfter)) {
                            $this->addDataInDatabase($productId, $sizeChartId, $storeId);
                        }
                    } else {
                        $this->addDataInDatabase($productId, $sizeChartId, $storeId);
                    }
                } else {
                    $this->addDataInDatabase($productId, $sizeChartId, $storeId);
                }
            } else {
                $this->addDataInDatabase($productId, $sizeChartId, $storeId);
            }
        }
    }

    /**
     * Add data in database
     * @param int $productId
     * @param int $sizeChartId
     * @param int $storeId
     * @return void
     */
    protected function addDataInDatabase($productId, $sizeChartId, $storeId)
    {
        if ($productId != 0) {
            if ($this->checkProductIssetSizeChart($productId, $storeId)) {
                $this->updateProduct($productId, $sizeChartId, $storeId);
            } else {
                $this->insertProduct($productId, $sizeChartId, $storeId);
            }
        }
    }

    /**
     * Check Product Isset Size Chart
     * @param int $productId
     * @param int $storeId
     * @return bool
     */
    protected function checkProductIssetSizeChart($productId,$storeId)
    {
        $check = true;
        $attributeId = $this->eavAttribute->getIdByCode(
            'catalog_product',
            'bss_sizechart'
        );
        $connection = $this->resource->getConnection();
        $table = $connection->getTableName('catalog_product_entity_varchar');
        $sql = $this->resourceSizeChart->buildSql($table, $productId, $attributeId, $storeId);
        $result = $connection->fetchRow($sql);
        if (!$result) {
            $check = false;
        }
        return $check;
    }

    /**
     * Update Product
     *
     * @param int $productId
     * @param int $sizeChartId
     * @param int $storeId
     * @return void
     */
    protected function updateProduct($productId,$sizeChartId,$storeId)
    {
        $attributeId = $this->eavAttribute->getIdByCode(
            'catalog_product',
            'bss_sizechart'
        );
        $connection = $this->resource->getConnection();
        $table = $connection->getTableName('catalog_product_entity_varchar');
        $bind = ['value' => (int)$sizeChartId];
        $where = [
            'entity_id = ?' => (int)$productId,
            'attribute_id = ?' => (int)$attributeId,
            'store_id = ?' => (int)$storeId
        ];
        $connection->update($table, $bind, $where);
    }

    /**
     * Insert Product
     *
     * @param int $productId
     * @param int $sizeChartId
     * @param int $storeId
     * @return void
     */
    protected function insertProduct($productId,$sizeChartId,$storeId)
    {
        $attributeId = $this->eavAttribute->getIdByCode(
            'catalog_product',
            'bss_sizechart'
        );
        $connection = $this->resource->getConnection();
        $table = $connection->getTableName('catalog_product_entity_varchar');
        $bind = [
            'value' => $sizeChartId,
            'entity_id' => (int)$productId,
            'attribute_id' => (int)$attributeId,
            'store_id' => (int)$storeId
        ];
        $this->resourceSizeChart->insertProduct($table, $bind);
    }

    /**
     * Treat String
     * @param array $stringDataDisplay
     * @return string
     */
    protected function treatString( $stringDataDisplay = [] )
    {
        $stringDisplay = implode(",", $stringDataDisplay);
        return $stringDisplay;
    }

    /**
     * Delete DataBase Product Assign SizeChart
     * @param string $table
     * @param array $where
     * @return void
     */
    protected function deleteDataBaseProductAssignSizeChart($table,$where)
    {
        $connection = $this->resource->getConnection();
        $connection->delete($table, $where);
    }

    /**
     * Reset Product Assign SizeChart
     * @param array $arrProduct
     * @param int $idSizeChart
     * @return void
     */
    protected function resetProductAssignSizeChart($arrProduct,$idSizeChart)
    {
        $connection = $this->resource->getConnection();
        $table = $connection->getTableName('catalog_product_entity_varchar');
        foreach ($arrProduct as $idProduct) {
            $where = [
                "value = ?" => $idSizeChart,
                "entity_id = ?" => $idProduct
            ];
            $this->deleteDataBaseProductAssignSizeChart($table, $where);
        }

    }

    /**
     * Save
     * @return \Magento\Backend\Model\View\Result\Redirect
     */
    public function execute()
    {
        $data = $this->getRequest()->getPostValue();

        /** @var \Magento\Backend\Model\View\Result\Redirect $resultRedirect */
        $resultRedirect = $this->resultRedirectFactory->create();
        $stringIdStore = null;

        $stringDisplay = isset(
            $data["display_popup"]
        ) ? $this->treatString($data["display_popup"]) : $this->getDefaultDisplay();
        $data["display_popup"] = $stringDisplay;
        $overwrite = isset($data["overwrite"]) ? "1" : "0";
        $arrProductIdSelected = [];
        if ($data) {
            try {
                $model = $this->sizeChartFactory->create();
                $modelRule = $this->ruleFactory->create();
                $id = $this->getRequest()->getParam('size_chart_value_id');
                if (isset($data['rule'])) {
                    $data['conditions'] = $data['rule']['conditions'];
                    unset($data['rule']);
                }

                $arrProductIdBySizeChart = [];
                if ($id) {
                    $model->load($id);
                    $idRule = $this->resourceRule->getRuleBySizeChartId($id);
                    if ($idRule) {
                        $modelRule->load($idRule);
                    }
                }

                $modelRule->loadPost($data);
                $modelRule->save();
                $arrProductIdBySizeChart = $this->getArrProductBySizeChart($id);
                $storeIdOld = $model->getIdStore();

                $model->setData($data);
                $model->save();
                $sizeChartId = $model->getId();
                $ruleId = $modelRule->getId();
                $this->resourceRule->insertDataRule($ruleId, $sizeChartId);
                $data['products'] = $modelRule->getProductIds();
                
                $dataProductSizeChart = [];
                foreach ($data['products'] as $productId) {
                    $dataProductSizeChart[] = [
                        'size_chart_id' => $sizeChartId,
                        'product_id' => $productId
                    ];
                }
                $whereProductSizeChart = [
                    'size_chart_id = ?' => $sizeChartId
                ];
                $this->resourceSizeChart->deleteMultiData('bss_product_sizechart', $whereProductSizeChart);
                $this->resourceSizeChart->insertMultiData('bss_product_sizechart', $dataProductSizeChart);

                $this->messageManager->addSuccessMessage(__('The size chart has been saved.'));
                $this->sessionFactory->create()->setFormData(false);
                if ($this->getRequest()->getParam('back')) {
                    return $resultRedirect->setPath('*/*/edit', [
                        'size_chart_value_id' => $model->getId(),
                        '_current' => true
                    ]);
                }
                return $resultRedirect->setPath('*/*/');
            } catch (\Magento\Framework\Exception\LocalizedException $e) {
                $this->messageManager->addErrorMessage($e->getMessage());
            } catch (\RuntimeException $e) {
                $this->messageManager->addErrorMessage($e->getMessage());
            } catch (\Exception $e) {
                $this->messageManager->addExceptionMessage($e, __($e->getMessage()));
            }

            $this->_getSession()->setFormData($data);
            return $resultRedirect->setPath('*/*/edit', [
                'size_chart_value_id' => $this->getRequest()->getParam('size_chart_value_id')
            ]);
        }
        return $resultRedirect->setPath('*/*/');
    }

    /**
     * Check rule admin
     * @return bool
     */
    protected function _isAllowed()
    {
        return $this->_authorization->isAllowed("Bss_SizeChart::save");
    }
}
