<?php
namespace Google\Cloud\Model\MediaStorage\File\Storage;

use Magento\Framework\DataObject;

class Gcs extends DataObject
{
    /**
     * @var \cAc\GcsWrapper\GoogleCloudStorage
     */
    private $client;
    /**
     * @var \Google\Cloud\Helper\Data
     */
    private $helper;
    /**
     * @var \Magento\MediaStorage\Helper\File\Storage\Database
     */
    private $storageHelper;
    /**
     * @var \Magento\MediaStorage\Helper\File\Media
     */
    private $mediaHelper;
    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;
    /**
     * @var \Magento\MediaStorage\Model\File\Storage\File
     */
    private $storageFile;

    /**
     * Collect errors during sync process
     *
     * @var string[]
     */
    private $errors = [];
    /**
     * @var \Magento\Framework\Filesystem\Io\File
     */
    private $io;


    /**
     * Gcs constructor.
     * @param \Google\Cloud\Helper\Data $helper
     * @param \Magento\MediaStorage\Helper\File\Media $mediaHelper
     * @param \Magento\MediaStorage\Helper\File\Storage\Database $storageHelper
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Magento\MediaStorage\Model\File\Storage\File $storageFile
     * @param \Magento\Framework\Filesystem\Io\File $io
     */
    public function __construct(
        \Google\Cloud\Helper\Data $helper,
        \Magento\MediaStorage\Helper\File\Media $mediaHelper,
        \Magento\MediaStorage\Helper\File\Storage\Database $storageHelper,
        \Psr\Log\LoggerInterface $logger,
		\Magento\MediaStorage\Model\File\Storage\File $storageFile,
        \Magento\Framework\Filesystem\Io\File $io
    ) {
        parent::__construct();

        $this->helper = $helper;
        $this->mediaHelper = $mediaHelper;
        $this->storageHelper = $storageHelper;
		$this->storageFile = $storageFile;
        $this->logger = $logger;
		$this->helper->getAccessKey();
		$json_key = $this->helper->getAccessKey();
        $key_array = json_decode( $json_key,true );
        $project = $key_array['project_id'];
        $this->io = $io;
        $this->client = new \cAc\GcsWrapper\GoogleCloudStorage($project, $json_key, $this->helper->getBucket());
    }

    /**
     * Initialisation
     *
     * @return $this
     */
    public function init() {
        return $this;
    }

    /**
     * Return storage name
     *
     * @return \Magento\Framework\Phrase
     */
    public function getStorageName() {
        return __('Google Cloud Storage');
    }
    /**
     * @param $filename
     * @return \Google\Cloud\Model\MediaStorage\File\Storage\Gcs
     */
    public function loadByFilename($filename) {
        try {
            $filePath = $this->storageHelper->getMediaBaseDir() . '/' . $filename;
            if(!file_exists($filePath)) {
                if($this->client->object_exists($filename)) {
                    $this->io->checkAndCreateFolder(dirname($filePath));
                    $object = $this->client->object_download($filePath);
                } else {
                    return $this;
                }
            }

            $this->setData('id', $filename);
            $this->setData('filename', $filename);
            $this->setData('content', (string) file_get_contents($filePath));
        } catch (\Exception $e) {
            $this->errors[] = $e->getMessage();
            $this->logger->critical($e->getMessage());
            $this->unsetData();
        }

        return $this;
    }

    /**
     * @return bool
     */
    public function hasErrors() {
        return !empty($this->errors);
    }

    /**
     * @return string[]
     */
    public function getErrors() {
        return $this->errors;
    }

    public function clear() {
        return $this;
    }
    public function exportDirectories($offset = 0, $count = 100) {
        return false;
    }
    public function importDirectories(array $dirs = []) {
        return $this;
    }
    public function getConnectionName() {
        return null;
    }

    public function exportFiles($offset = 0, $count = 100) {
        return $this->storageFile->exportFiles($offset,$count);
    }

    public function importFiles(array $files = [])
    {
        foreach( $files as $file ) {
            $this->saveFile($file['directory'].'/'.$file['filename']);
        }
        return $this;
    }

    public function saveFile($filename, $newName = null, $overwrite = false)
    {
        if(empty($filename)) {
            return $this;
        }

        if(!empty($newName) && $this->helper->getIsAnExcludedFolder(dirname($newName))) {
            return $this;
        } elseif(empty($newName) && $this->helper->getIsAnExcludedFolder(dirname($filename))) {
            return $this;
        }

        if(empty($newName)) {
            $newName = $filename;
        }

        try {
            if($overwrite || !$this->fileExists($filename)) {
                $this->client->bucket_upload_object(
                    $filename,
                    $this->storageHelper->getMediaBaseDir(),
                    $newName,
                    false,
                    "publicRead"
                );
            }
        } catch (\Exception $e) {
            $this->errors[] = $e->getMessage();
            $this->logger->critical($e->getMessage());
        }
        return $this;
    }

    public function fileExists($filename)
    {
        return $this->client->object_exists($filename);
    }

    public function copyFile($oldFilePath, $newFilePath)
    {
        try {
            $this->saveFile($oldFilePath, $newFilePath);
        } catch (\Exception $e) {
            $this->errors[] = $e->getMessage();
            $this->logger->critical($e->getMessage());
        }
        return $this;
    }

    public function renameFile($oldFilePath, $newFilePath)
    {
        try {
            $this->saveFile($oldFilePath, $newFilePath);
            $this->deleteFile($oldFilePath);
        } catch (\Exception $e) {
            $this->errors[] = $e->getMessage();
            $this->logger->critical($e->getMessage());
        }
        return $this;
    }

    /**
     * @param string $path
     * @return $this
     */
    public function deleteFile($path)
    {
        try {
            if($this->fileExists($path)) {
                $this->client->object_delete($path);
            }
        } catch (\Exception $e) {
            $this->errors[] = $e->getMessage();
            $this->logger->critical($e->getMessage());
        }
        return $this;
    }

    public function getSubdirectories($path)
    {
        $subdirectories = [];
        $prefix = $this->storageHelper->getMediaRelativePath($path);
        $prefix = rtrim($prefix, '/') . '/';
        return $subdirectories;
    }

    public function getDirectoryFiles($path)
    {
        $files = [];
        $prefix = $this->storageHelper->getMediaRelativePath($path);
        $prefix = rtrim($prefix, '/') . '/';
        return $files;
    }

    public function deleteDirectory($path)
    {
        $mediaRelativePath = $this->storageHelper->getMediaRelativePath($path);
        $prefix = rtrim($mediaRelativePath, '/') . '/';

        try {
            $result = $this->client->bucket_get_objects([
                'bucket' => $this->helper->getBucket(),
                'prefix' => $prefix
            ]);

            foreach($result as $object) {
                $object->delete();
            }
        } catch (\Exception $e) {
            $this->errors[] = $e->getMessage();
            $this->logger->critical($e->getMessage());
        }
        return $this;
    }

    public function deleteFolder($path) {
        return $this->deleteDirectory($path);
    }

}
