<?php
namespace Google\Cloud\Console\Command;

use Magento\Config\Model\Config\Factory;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class StorageSyncCommand extends \Symfony\Component\Console\Command\Command
{
    private $client;

    /**
     * @var \Magento\Config\Model\Config\Factory
     */
    private $configFactory;
    /**
     * @var \Magento\Framework\App\State
     */
    private $state;
    /**
     * @var \Google\Cloud\Helper\Data
     */
    private $helper;
    /**
     * @var \Magento\MediaStorage\Helper\File\Storage
     */
    private $coreFileStorage;
    /**
     * @var \Magento\MediaStorage\Helper\File\Storage\Database
     */
    private $storageHelper;
    /**
     * @var \Google\Cloud\Logger\Logger
     */
    private $logger;

    /**
     * StorageSyncCommand constructor.
     * @param \Magento\Framework\App\State $state
     * @param \Magento\Config\Model\Config\Factory $configFactory
     * @param \Magento\MediaStorage\Helper\File\Storage\Database $storageHelper
     * @param \Magento\MediaStorage\Helper\File\Storage $coreFileStorage
     * @param \Google\Cloud\Helper\Data $helper
     * @param \Google\Cloud\Logger\Logger $logger
     */
    public function __construct(
        \Magento\Framework\App\State $state,
        Factory $configFactory,
        \Magento\MediaStorage\Helper\File\Storage\Database $storageHelper,
        \Magento\MediaStorage\Helper\File\Storage $coreFileStorage,
        \Google\Cloud\Helper\Data $helper,
        \Google\Cloud\Logger\Logger $logger
    ) {
        $this->state = $state;
        $this->configFactory = $configFactory;
        $this->coreFileStorage = $coreFileStorage;
        $this->helper = $helper;
        $this->storageHelper = $storageHelper;
        $this->logger = $logger;
        parent::__construct();
    }

    protected function configure()
    {
        $this->setName('gcs:storage:sync')
            ->setDescription('Sync all of your media files over to GCS.')
            ->setDefinition($this->getOptionsList());
        parent::configure();
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $errors = $this->helper->validate();
        if ($errors) {
            $output->writeln('<error>' . implode('</error>' . PHP_EOL .  '<error>', $errors) . '</error>');
            return;
        }

        try {
        	$json_key = $this->helper->getAccessKey();
        	$key_array = json_decode($json_key,true);
        	$project = $key_array['project_id'];
        	$this->client = new \cAc\GcsWrapper\GoogleCloudStorage(
				$project,
        		$json_key,
        		$this->helper->getBucket()
        	);
        } catch (\Exception $e) {
            $output->writeln(sprintf('<error>%s</error>', $e->getMessage()));
            return;
        }
        if (!$this->client->bucket_exists()) {
            $output->writeln('<error>The GCS credentials you provided did not work. Please review your details and try again. You can do so using our config script.</error>');
            return;
        }
        /*if ($this->coreFileStorage->getCurrentStorageCode() == \Google\Cloud\Model\MediaStorage\File\Storage::STORAGE_MEDIA_GCS) {
            $output->writeln('<error>You are already using GCS as your media file storage backend!</error>');
            return;
        }*/

        $output->writeln(sprintf('Uploading files to use GCS...'));
        if ($this->coreFileStorage->getCurrentStorageCode() == \Google\Cloud\Model\MediaStorage\File\Storage::STORAGE_MEDIA_FILE_SYSTEM) {
            try {
                $directoryToSync = $this->storageHelper->getMediaBaseDir();
                $mediaFiles = glob($directoryToSync . '/*', GLOB_MARK);
                foreach($mediaFiles as $file) {
                    if(!$this->helper->getIsAnExcludedFolder(str_replace($directoryToSync . '/', '', $file))) {
                        $this->client->bucket_upload_directory(
                            $file,
                            $directoryToSync . '/',
                            false,
                            "publicRead"
                        );
                    }
                }
            } catch (\Exception $e) {
                $output->writeln(sprintf('<error>%s</error>', $e->getMessage()));
            }
        } else {
            $sourceModel = $this->coreFileStorage->getStorageModel();
            $destinationModel = $this->coreFileStorage->getStorageModel(\Google\Cloud\Model\MediaStorage\File\Storage::STORAGE_MEDIA_GCS);
            $offset = 0;
            while (($files = $sourceModel->exportFiles($offset, 1)) !== false) {
                foreach ($files as $file) {
                    $output->writeln(sprintf('Uploading %s to use GCS.', $file['directory'] . '/' . $file['filename']));
                }
                $destinationModel->importFiles($files);
                $offset += count($files);
            }
        }

        $errors = $this->client->getErrors();
        if(!empty($errors)) {
            foreach($errors as $error) {
                $output->writeln(sprintf('<error>%s</error>', $error));
            }
            $output->writeln('');
        }

        $logs = $this->client->getLogs();
        if(!empty($logs)) {
            foreach($logs as $log) {
                $this->logger->info($log);
                $output->writeln(sprintf('<info>%s<info>', $log));
            }
            $output->writeln('');
        }

        $output->writeln(sprintf('Finished uploading files to use GCS.'));

        /**
         * Remove all synced files excluding original catalog/product images
         */
//        $output->writeln(sprintf('Start to remove synced files...'));
//        $logs = $this->removeFiles($directoryToSync, $directoryToSync . '/');
//        if(!empty($logs)) {
//            foreach($logs as $log) {
//                echo PHP_EOL;
//                print_r($log);
//                echo PHP_EOL;
//            }
//        }

        if ($input->getOption('enable')) {
            $output->writeln('Updating configuration to use GCS.');

            try {
                $this->state->getAreaCode();
            } catch (Exception $e) {
                $this->state->setAreaCode('adminhtml');
            }

            $config = $this->configFactory->create();
            $config->setDataByPath('system/media_storage_configuration/media_storage', \Google\Cloud\Model\MediaStorage\File\Storage::STORAGE_MEDIA_GCS);
            $config->save();
            $output->writeln(sprintf('<info>Magento now uses GCS for its file backend storage.</info>'));
        }
    }

//    public function removeFiles($directoryToRemove, $replaceFromSource) {
//        $results = [];
//        $dir_files = glob( $directoryToRemove . '/*', GLOB_MARK );
//        foreach( $dir_files as $file ) {
//            $file = realpath($file);
//            if( !is_dir( $file )) {
//                try {
//                    if( $this->client->object_exists(str_replace($replaceFromSource, '', $file)) && !$this->helper->getIsCatalogProductOriginalImage($file)) {
//                        unlink($file);
//                        $results['success'][] = 'Deleted file ' . $file;
//                    }
//                } catch( \Exception $e ) {
//                    $this->logger->error("Cannot remove file $file. Exception: " . $e->getMessage());
//                    $results[]['error'] = "Cannot remove file $file. Exception: " . sprintf('<error>%s</error>', $e->getMessage());
//                }
//            }
//
//            if(is_dir($file) && $this->helper->is_dir_empty($file)) {
//                try {
//                    rmdir($file);
//                    $results['success'][] = 'Deleted empty folder ' . $file;
//                } catch (Exception $e) {
//                    $this->logger->error("Cannot remove dir $file. Exception: " . $e->getMessage());
//                    $results['error'][] = "Cannot remove dir $file. Exception: " . sprintf('<error>%s</error>', $e->getMessage());
//                }
//            } else {
//                $results = array_merge($results, $this->removeFiles($file, $replaceFromSource));
//            }
//        }
//
//        return $results;
//    }

    public function getOptionsList()
    {
        return [new InputOption('enable', null, InputOption::VALUE_NONE, 'use GCS as Magento file storage backend'),];
    }

}
